package com.els.tso.workflow.service.impl;


import com.els.tso.workflow.exception.AuditException;
import com.els.tso.workflow.service.WorkFlowService;
import org.activiti.engine.*;
import org.activiti.engine.identity.Group;
import org.activiti.engine.repository.ProcessDefinition;
import org.activiti.engine.runtime.ProcessInstance;
import org.activiti.engine.task.IdentityLink;
import org.activiti.engine.task.Task;
import org.activiti.engine.task.TaskQuery;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Component
public class WorkFlowServiceImpl implements WorkFlowService {

    @Resource
    protected TaskService taskService;

    @Resource
    protected RuntimeService runtimeService;

    @Resource
    protected HistoryService historyService;
    @Resource
    protected IdentityService identityService;
    @Resource
    protected RepositoryService repositoryService;


    @Transactional(rollbackFor = AuditException.class)
    @Override
    public void completePersonalTask(String taskId, String userId, String isPass, String desc) {
        // 1、查询任务,并校验任务
        Task task = queryTaskAndValid(taskId);

        // 2、校验用户是否有权限
        validUser(task, userId);

        // 3、当办理人为空时组用户领取任务
        if (StringUtils.isBlank(task.getAssignee())) {
            taskService.claim(taskId, userId);
        }
        // 4、获取当前的流程节点
        ProcessInstance currentProcess = runtimeService.createProcessInstanceQuery()
            .processInstanceId(task.getProcessInstanceId()).includeProcessVariables().singleResult();
        if (currentProcess == null || currentProcess.isEnded()) {
            throw new AuditException("该审批流程已经结束");
        }
        // 流程禁用的处理
        String processDefinitionKey = (String) currentProcess.getProcessVariables().get("processDefinitionKey");
        String businessKey = currentProcess.getBusinessKey();
        List<ProcessDefinition> processDefinitions = repositoryService.createProcessDefinitionQuery()
            .processDefinitionKey(processDefinitionKey).list();
        if (CollectionUtils.isEmpty(processDefinitions)) {
            throw new AuditException("未找到该模块编码,请检查数据!");
        }
        ProcessDefinition definition = repositoryService.createProcessDefinitionQuery()
            .processDefinitionKey(processDefinitionKey).list().get(processDefinitions.size() - 1);
        if (repositoryService.isProcessDefinitionSuspended(definition.getId())) {

            ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processDefinitionKey(processDefinitionKey).processInstanceBusinessKey(businessKey).singleResult();
            runtimeService.deleteProcessInstance(processInstance.getId(), "审批流已停用");
        } else {

            // 添加备注信息
            taskService.addComment(taskId, task.getProcessInstanceId(), desc);
            // 办理任务
            taskService.complete(taskId);

        }
    }

    private List<String> getUserIdByTask(Task task) {
        List<String> userIdList = new ArrayList<>();
        userIdList.add(task.getAssignee());

        List<IdentityLink> identityLinks = this.taskService.getIdentityLinksForTask(task.getId());

        List<String> candidateUsers = identityLinks.stream()
            .map(IdentityLink::getUserId).collect(Collectors.toList());

        userIdList.addAll(candidateUsers);

        List<String> userInGroups = identityLinks.stream()
            .map(IdentityLink::getGroupId)
            .filter(StringUtils::isNotBlank)
            .collect(Collectors.toList());

        userIdList.addAll(userInGroups);
        return userIdList;
    }


    private void validUser(Task task, String userId) {
        if (StringUtils.isEmpty(task.getAssignee())) {
            ArrayList<String> list = new ArrayList<>();
            for (IdentityLink identityLink : taskService.getIdentityLinksForTask(task.getId())) {
                if (StringUtils.isNotEmpty(identityLink.getUserId())) {
                    list.add(identityLink.getUserId());
                }
            }
            if (!CollectionUtils.isEmpty(list)) {
                if (!list.contains(userId)) {
                    throw new AuditException("没有权限执行该审核任务");
                }
            } else {
                // 角色权限判断
                List<Group> groupList = this.identityService.createGroupQuery().groupMember(userId).list();
                List<String> groupIdList = new ArrayList<>();
                for (Group group : groupList) {
                    groupIdList.add(group.getId());
                }
                List<IdentityLink> links = taskService.getIdentityLinksForTask(task.getId());
                List<String> linkGroupIds = new ArrayList<>();
                for (IdentityLink link : links) {
                    linkGroupIds.add(link.getGroupId());
                }

                if (!CollectionUtils.containsAny(groupIdList, linkGroupIds)) {
                    throw new AuditException("没有权限执行该审核任务");

                }
            }
        } else if (!userId.equals(task.getAssignee())) {
            throw new AuditException("没有权限执行该审核任务");
        }

    }

    private Task queryTaskAndValid(String taskId) {
        // 获取查询对象
        TaskQuery taskQuery = taskService.createTaskQuery()
            .includeTaskLocalVariables().taskId(taskId);

        Task task = taskQuery.singleResult();
        if (task == null) {
            throw new AuditException("该任务id不存在");
        }

        if (task.isSuspended()) {
            throw new AuditException("该审批任务已经暂停");
        }

        return task;
    }

    @Override
    public ProcessInstance stopProcess(String processDefinitionKey, String businessKey) {
        if (StringUtils.isBlank(processDefinitionKey) || StringUtils.isBlank(businessKey)) {
            throw new AuditException("流程key，业务key，发起人不允许为空");
        }

        List<ProcessInstance> processInstances = runtimeService.createProcessInstanceQuery()
            .processDefinitionKey(processDefinitionKey).processInstanceBusinessKey(businessKey).list();
        if (CollectionUtils.isEmpty(processInstances)) {
            throw new AuditException("未找到该模块编码或业务编码,请检查数据!");
        }
        for (ProcessInstance processInstance : processInstances) {
            runtimeService.deleteProcessInstance(processInstance.getId(), "已作废");
            return processInstance;
        }
        return null;
    }

    @Override
    public ProcessInstance startProcess(String processDefinitionKey, String businessKey,
                                        Map<String, Object> processVariable) {
        if (StringUtils.isBlank(processDefinitionKey) || StringUtils.isBlank(businessKey)) {
            throw new AuditException("流程key，业务key，发起人不允许为空");
        }
        ProcessInstance processInstance = null;
        List<ProcessDefinition> processDefinitions = repositoryService.createProcessDefinitionQuery()
            .processDefinitionKey(processDefinitionKey).list();
        if (CollectionUtils.isEmpty(processDefinitions)) {
            throw new AuditException("未找到该模块编码,请检查数据!");
        }
        ProcessDefinition definition = repositoryService.createProcessDefinitionQuery()
            .processDefinitionKey(processDefinitionKey).list().get(processDefinitions.size() - 1);
        if (repositoryService.isProcessDefinitionSuspended(definition.getId())) {
        } else {
            ProcessInstance currentProcess = runtimeService.createProcessInstanceQuery()
                .processDefinitionKey(processDefinitionKey).processInstanceBusinessKey(businessKey).singleResult();

            if (currentProcess != null) {
                throw new AuditException("业务流程[" + processDefinitionKey + "][" + businessKey + "]，已经启动，无法重复启动");
            }
            processVariable.put("startTime", new Date());
            processInstance = runtimeService.startProcessInstanceByKey(processDefinitionKey, businessKey,
                processVariable);
            List<Task> list = this.taskService.createTaskQuery().processInstanceId(processInstance.getId()).list();

        }
        return processInstance;
    }


    private List<Task> distinctAndSort(List<Task> allTask) {
        Set<Task> taskSet = new TreeSet<Task>((t1, t2) -> {
            return t1.getId().compareTo(t2.getId());
        });

        taskSet.addAll(allTask);

        allTask.clear();
        allTask.addAll(taskSet);

        allTask.sort((t1, t2) -> {

            if (t1.getCreateTime() == null) {
                return -1;
            }

            if (t2.getCreateTime() == null) {
                return 1;
            }
            return t2.getCreateTime().compareTo(t1.getCreateTime());
        });
        return allTask;
    }

}
