package com.els.tso.workflow.service.impl;


import com.els.tso.workflow.constant.ActivitiConstant;
import com.els.tso.workflow.dto.TaskOutDTO;
import com.els.tso.workflow.enumeration.AuditCommentTypeEnum;
import com.els.tso.workflow.enumeration.AuditStatusEnum;
import com.els.tso.workflow.model.AuditResult;
import com.els.tso.workflow.service.ActivitiOptService;
import org.activiti.bpmn.model.BpmnModel;
import org.activiti.bpmn.model.FlowNode;
import org.activiti.bpmn.model.SequenceFlow;
import org.activiti.engine.*;
import org.activiti.engine.history.HistoricActivityInstance;
import org.activiti.engine.history.HistoricProcessInstance;
import org.activiti.engine.history.HistoricTaskInstance;
import org.activiti.engine.history.ProcessInstanceHistoryLog;
import org.activiti.engine.identity.Group;
import org.activiti.engine.identity.User;
import org.activiti.engine.runtime.Execution;
import org.activiti.engine.runtime.ProcessInstance;
import org.activiti.engine.task.Comment;
import org.activiti.engine.task.Task;
import org.activiti.engine.task.TaskInfo;
import org.activiti.image.ProcessDiagramGenerator;
import org.activiti.image.impl.DefaultProcessDiagramGenerator;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.io.InputStream;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

/**
 * activiti相关操作
 *
 * @author daipengjie
 * @since 2020/10/09
 */
@Service
public class ActivitiOptServiceImpl implements ActivitiOptService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ActivitiOptServiceImpl.class);

    @Autowired
    protected FormService formService;
    @Autowired
    private RepositoryService repositoryService;
    @Autowired
    private HistoryService historyService;
    @Autowired
    private RuntimeService runtimeService;
    @Autowired
    private TaskService taskService;
    @Autowired
    private IdentityService identityService;

    @Override
    public AuditResult submit(String processDefinitionKey, String userId, String businessKey, String auditOpinion, Map<String, Object> variables) {

        Assert.hasLength(processDefinitionKey, ActivitiConstant.Message.PROCESS_DEFINITION_KEY_CANNOT_BLANK);
        AuditResult result = new AuditResult();
        identityService.setAuthenticatedUserId(userId);
        ProcessInstance processInstance = runtimeService.startProcessInstanceByKey(processDefinitionKey, businessKey, variables);
        if (StringUtils.isNotBlank(auditOpinion)) {
            processInstance.getProcessVariables().put(ActivitiConstant.START_AUDIT_OPINION_VARIABLE_KEY, auditOpinion);
        }
        result.setProcessRootId(processInstance.getProcessInstanceId());
        result.setAuditStatus(AuditStatusEnum.IN_AUDIT.getValue());
        result.setSubmitUser(userId);

        final List<Task> tasks = getProcessTasks(processInstance.getProcessInstanceId());

        if (tasks != null) {
            Set<String> auditUserList = tasks.parallelStream().map(Task::getAssignee).collect(Collectors.toSet());
            result.setNextAuditUserList(auditUserList);
        }
        return result;
    }

    @Override
    public AuditResult complete(String taskId, String userId, String auditOpinion, Map<String, Object> variables) {

        Assert.hasLength(taskId, ActivitiConstant.Message.TASK_ID_CANNOT_BLANK);
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        //任务校验
        Assert.notNull(task, ActivitiConstant.Message.TASK_NOT_EXIST);
        Assert.isTrue((Objects.equals(userId, task.getAssignee())), ActivitiConstant.Message.TASK_NOT_BELONG_CURRENT_USER);

        String processInstanceId = task.getProcessInstanceId();

        taskService.claim(taskId, userId);
        if (StringUtils.isNotBlank(auditOpinion)) {
            taskService.addComment(taskId, processInstanceId, AuditCommentTypeEnum.OPINION.getType(), auditOpinion);
        }
        taskService.addComment(taskId, processInstanceId, AuditCommentTypeEnum.STATE.getType(), AuditStatusEnum.AUDIT_PASS.getDesc());
        taskService.complete(taskId, variables);

        List<Task> tasks = getProcessTasks(processInstanceId);

        AuditResult result = new AuditResult();
        result.setProcessRootId(processInstanceId);
        result.setSubmitUser(userId);
        //为空则表示审批通过且流程结束
        if (tasks == null || tasks.size() == 0) {
            HistoricProcessInstance processInstance = historyService.createHistoricProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();
            String startUserId = processInstance.getStartUserId();
            result.setAuditStatus(AuditStatusEnum.AUDIT_PASS.getValue());
            result.setNextAuditUserList(Collections.singleton(startUserId));
        } else {
            result.setAuditStatus(AuditStatusEnum.IN_AUDIT.getValue());
            Set<String> auditUserList = tasks.parallelStream().map(Task::getAssignee).collect(Collectors.toSet());
            result.setNextAuditUserList(auditUserList);
        }
        return result;
    }

    @Override
    public AuditResult refuse(String taskId, String userId, String auditOpinion) {

        Assert.hasLength(taskId, ActivitiConstant.Message.TASK_ID_CANNOT_BLANK);
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        Assert.notNull(task, ActivitiConstant.Message.TASK_NOT_EXIST);
        Assert.isTrue((Objects.equals(userId, task.getAssignee())), ActivitiConstant.Message.TASK_NOT_BELONG_CURRENT_USER);
        String processInstanceId = task.getProcessInstanceId();
        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();

        String startUserId = processInstance.getStartUserId();
        taskService.addComment(taskId, processInstanceId, AuditCommentTypeEnum.STATE.getType(), AuditStatusEnum.AUDIT_NOPASS.getDesc());
        taskService.addComment(taskId, processInstanceId, AuditCommentTypeEnum.OPINION.getType(), auditOpinion);
        runtimeService.deleteProcessInstance(processInstanceId, auditOpinion);
        AuditResult result = new AuditResult();
        result.setProcessRootId(processInstanceId);
        result.setSubmitUser(userId);
        result.setAuditStatus(AuditStatusEnum.AUDIT_NOPASS.getValue());
        result.setNextAuditUserList(Collections.singleton(startUserId));
        return result;
    }

    @Override
    public AuditResult reject(String taskId, String userId, String auditOpinion) {
        return null;
    }


    @Override
    public AuditResult cancelSubmit(String userId, String processInstanceId, String auditOpinion) {
        //查询代办
        List<Task> taskList = taskService.createTaskQuery().processInstanceId(processInstanceId).list();
        HistoricProcessInstance historicProcessInstance = historyService.createHistoricProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();
        Assert.notNull(historicProcessInstance, ActivitiConstant.Message.PROCESS_INSTANCE_NOT_EXIST);
        //获取发起人
        String startUserId = historicProcessInstance.getStartUserId();
        Assert.isTrue(Objects.equals(userId, startUserId), ActivitiConstant.Message.TASK_NOT_BELONG_CURRENT_USER);
        identityService.setAuthenticatedUserId(userId);
        runtimeService.deleteProcessInstance(processInstanceId, auditOpinion);
        //封装返回
        return getAuditResult(taskList, userId, processInstanceId, AuditStatusEnum.UN_AUDIT);
    }


    /**
     * @param userId
     * @param taskId       待撤回的任务id
     * @param auditOpinion
     */
    @Override
    public AuditResult cancelComplete(String userId, String taskId, String auditOpinion) {


        //任务校验
        Assert.hasLength(taskId, ActivitiConstant.Message.TASK_ID_CANNOT_BLANK);
        HistoricTaskInstance historicTaskInstance = historyService.createHistoricTaskInstanceQuery().taskId(taskId).singleResult();
        Assert.notNull(historicTaskInstance, ActivitiConstant.Message.TASK_NOT_EXIST);
        Assert.isTrue((Objects.equals(userId, historicTaskInstance.getAssignee())), ActivitiConstant.Message.TASK_NOT_BELONG_CURRENT_USER);

        String processInstanceId = historicTaskInstance.getProcessInstanceId();
        String myActivityId = null;
        List<HistoricActivityInstance> haiList = historyService.createHistoricActivityInstanceQuery()
            .executionId(historicTaskInstance.getExecutionId()).finished().list();
        for (HistoricActivityInstance hai : haiList) {
            if (taskId.equals(hai.getTaskId())) {
                myActivityId = hai.getActivityId();
                break;
            }
        }
        Assert.notNull(myActivityId, ActivitiConstant.Message.TASK_NOT_EXIST);
        BpmnModel bpmnModel = repositoryService.getBpmnModel(historicTaskInstance.getProcessDefinitionId());
        FlowNode myFlowNode = (FlowNode) bpmnModel.getMainProcess().getFlowElement(myActivityId);
        List<Task> list = taskService.createTaskQuery().executionId(historicTaskInstance.getExecutionId()).list();

        Execution execution = runtimeService.createExecutionQuery().executionId(historicTaskInstance.getExecutionId()).singleResult();
        String activityId = execution.getActivityId();
        FlowNode flowNode = (FlowNode) bpmnModel.getMainProcess().getFlowElement(activityId);
        //记录原活动方向
        List<SequenceFlow> oriSequenceFlows = new ArrayList<>(flowNode.getOutgoingFlows());
        //清理活动方向
        flowNode.getOutgoingFlows().clear();
        //建立新方向
        List<SequenceFlow> newSequenceFlowList = new ArrayList<>();
        SequenceFlow newSequenceFlow = new SequenceFlow();
        newSequenceFlow.setSourceFlowElement(flowNode);
        newSequenceFlow.setTargetFlowElement(myFlowNode);
        newSequenceFlowList.add(newSequenceFlow);
        flowNode.setOutgoingFlows(newSequenceFlowList);
        //查询代办
        List<Task> taskList = taskService.createTaskQuery().processInstanceId(processInstanceId).list();
        list.forEach((taskTmp) -> {
            identityService.setAuthenticatedUserId(userId);
            if (StringUtils.isNotBlank(auditOpinion)) {
                taskService.addComment(taskTmp.getId(), taskTmp.getProcessInstanceId(), AuditCommentTypeEnum.OPINION.getType(), auditOpinion);
            }
            taskService.addComment(taskTmp.getId(), taskTmp.getProcessInstanceId(), AuditCommentTypeEnum.STATE.getType(), AuditStatusEnum.AUDIT_CANCEL.getDesc());
            //完成任务
            taskService.complete(taskTmp.getId());
        });
        //恢复原方向
        flowNode.setOutgoingFlows(oriSequenceFlows);
        //封装返回
        return getAuditResult(taskList, userId, processInstanceId, AuditStatusEnum.IN_AUDIT);
    }

    @Override
    public boolean isFlowNodeJustBefore(String taskId) {

        Assert.hasLength(taskId, ActivitiConstant.Message.TASK_ID_CANNOT_BLANK);
        HistoricTaskInstance historicTaskInstance = historyService.createHistoricTaskInstanceQuery().taskId(taskId).singleResult();
        Assert.notNull(historicTaskInstance, ActivitiConstant.Message.TASK_NOT_EXIST);
        List<HistoricActivityInstance> haiList = historyService.createHistoricActivityInstanceQuery()
            .executionId(historicTaskInstance.getExecutionId()).list();
        String myActivityId = null;
        for (HistoricActivityInstance hai : haiList) {
            if (taskId.equals(hai.getTaskId())) {
                myActivityId = hai.getActivityId();
                break;
            }
        }
        BpmnModel bpmnModel = repositoryService.getBpmnModel(historicTaskInstance.getProcessDefinitionId());
        Execution execution = runtimeService.createExecutionQuery().executionId(historicTaskInstance.getExecutionId()).singleResult();
        String activityId = execution.getActivityId();
        FlowNode myFlowNode = (FlowNode) bpmnModel.getMainProcess().getFlowElement(myActivityId);
        FlowNode flowNode = (FlowNode) bpmnModel.getMainProcess().getFlowElement(activityId);
        AtomicBoolean check = new AtomicBoolean(false);
        myFlowNode.getOutgoingFlows().forEach(outFlow -> {
            flowNode.getIncomingFlows().forEach(inFlow -> {
                if (outFlow == inFlow) {
                    check.set(true);
                }
            });
        });
        return check.get();
    }

    @Override
    public boolean isJustSubmit(String processInstanceId) {

        Assert.hasLength(processInstanceId, ActivitiConstant.Message.PROCESS_INSTANCE_ID_CANNOT_BLANK);
        HistoricProcessInstance historicProcessInstance = historyService.createHistoricProcessInstanceQuery().processInstanceId(processInstanceId).singleResult();
        Assert.notNull(historicProcessInstance, ActivitiConstant.Message.PROCESS_INSTANCE_NOT_EXIST);
        BpmnModel bpmnModel = repositoryService.getBpmnModel(historicProcessInstance.getProcessDefinitionId());
        FlowNode myFlowNode = (FlowNode) bpmnModel.getMainProcess().getInitialFlowElement();
        AtomicBoolean allCheck = new AtomicBoolean(true);
        List<Task> list = taskService.createTaskQuery().processInstanceId(processInstanceId).list();
        for (Task task : list) {
            Execution execution = runtimeService.createExecutionQuery().executionId(task.getExecutionId()).singleResult();
            FlowNode flowNode = (FlowNode) bpmnModel.getMainProcess().getFlowElement(execution.getActivityId());
            AtomicBoolean check = new AtomicBoolean(false);
            myFlowNode.getOutgoingFlows().forEach(outFlow -> {
                flowNode.getIncomingFlows().forEach(inFlow -> {
                    if (outFlow == inFlow) {
                        check.set(true);
                    }
                });
            });
            if (!check.get()) {
                allCheck.set(false);
                break;
            }
        }

        return allCheck.get();
    }

    @Override
    public InputStream showDiagram(String processInstanceId) {

        return this.getProcessDiagram(processInstanceId);
    }

    @Override
    public String createUser(String userId, String userName) {

        User userOld = identityService.createUserQuery().userId(userId).singleResult();
        if (userOld != null) {
            identityService.deleteUser(userId);
        }
        User user = identityService.newUser(userId);
        user.setFirstName(userName);
        identityService.saveUser(user);
        return userId;
    }

    @Override
    public String createUserGroup(String groupId, String groupName, String userId) {

        Group group = identityService.createGroupQuery().groupId(groupId).singleResult();
        if (group == null) {
            Group groupNew = identityService.newGroup(groupId);
            groupNew.setName(groupName);
            identityService.saveGroup(groupNew);
        }
        if (identityService.createUserQuery().userId(userId).singleResult() != null) {
            identityService.createMembership(userId, groupId);
        } else {
            LOGGER.warn(ActivitiConstant.Message.USER_ID_NOT_EXIST_AND_IGNORE_GROUP_CREATE);
        }
        return groupId;
    }

    private List<Task> getProcessTasks(String processInstanceId) {

        return taskService.createTaskQuery().processInstanceId(processInstanceId).list();
    }

    @Override
    public List<TaskOutDTO> todoList(String userId) {

        return todoPageList(userId, 1, Integer.MAX_VALUE);
    }

    @Override
    public List<TaskOutDTO> todoPageList(String userId, int pageNum, int pageSize) {

        final List<Task> tasks = taskService.createTaskQuery().taskAssignee(userId).orderByTaskCreateTime().desc().listPage((pageNum - 1) * pageSize, pageSize);
        return tasks.parallelStream().map(this::converse).collect(Collectors.toList());
    }

    @Override
    public List<TaskOutDTO> auditHisList(String processInstanceId) {

        ProcessInstanceHistoryLog historyLog = historyService.createProcessInstanceHistoryLogQuery(processInstanceId).includeActivities().includeComments().singleResult();

        Assert.notNull(historyLog, ActivitiConstant.Message.PROCESS_INSTANCE_NOT_EXIST);
        List<TaskOutDTO> list = new ArrayList<>();
        TaskOutDTO start = new TaskOutDTO();
        list.add(start);
        start.setAssignee(historyLog.getStartUserId());
        start.setCreateTime(historyLog.getStartTime());
        start.setState(ActivitiConstant.START_STATE);
        historyLog.getHistoricData().forEach(log -> {
            if (log instanceof HistoricActivityInstance) {
                HistoricActivityInstance historicActivityInstance = (HistoricActivityInstance) log;
                if (ActivitiConstant.USER_TASK.equals(historicActivityInstance.getActivityType()) && historicActivityInstance.getEndTime() != null) {
                    TaskOutDTO outDTO = new TaskOutDTO();
                    list.add(outDTO);
                    outDTO.setAssignee(historicActivityInstance.getAssignee());
                    outDTO.setCreateTime(historicActivityInstance.getEndTime());
                    outDTO.setId(historicActivityInstance.getTaskId());
                    if (StringUtils.isNotBlank(historicActivityInstance.getDeleteReason())) {
                        outDTO.setOpinion(historicActivityInstance.getDeleteReason());
                        outDTO.setState(AuditStatusEnum.AUDIT_CANCEL.getDesc());
                    }
                }
            }
        });
        list.forEach(dto -> {
            historyLog.getHistoricData().forEach(log -> {
                if (log instanceof Comment) {
                    Comment comment = (Comment) log;
                    if (dto.getId() != null && dto.getId().equals(comment.getTaskId())) {
                        if (AuditCommentTypeEnum.STATE.getType().equals(comment.getType())) {
                            dto.setState(comment.getFullMessage());
                        } else {
                            dto.setOpinion(comment.getFullMessage());
                        }
                    }
                }
            });
        });
        return list;
    }

    private TaskOutDTO converse(Task task) {

        TaskOutDTO outDTO = new TaskOutDTO();
        outDTO.setId(task.getId());
        outDTO.setAssignee(task.getAssignee());
        outDTO.setCreateTime(task.getCreateTime());
        outDTO.setDelegationState(task.getDelegationState());
        outDTO.setDescription(task.getDescription());
        outDTO.setDueDate(task.getDueDate());
        outDTO.setFormKey(task.getFormKey());
        outDTO.setName(task.getName());
        outDTO.setParentTaskId(task.getParentTaskId());
        outDTO.setPriority(task.getPriority());
        outDTO.setProcessInstanceId(task.getProcessInstanceId());
        outDTO.setOwner(task.getOwner());
        final String businessKey = runtimeService.createProcessInstanceQuery().processInstanceId(task.getProcessInstanceId()).singleResult().getBusinessKey();
        outDTO.setBusinessKey(businessKey);
        return outDTO;
    }

    @Override
    public long todoListCount(String userId) {

        return taskService.createTaskQuery().taskAssignee(userId).count();
    }

    /**
     * Get Process instance diagram,jpeg格式
     */
    public InputStream getProcessDiagram(String processInstanceId) {

        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
            .processInstanceId(processInstanceId).singleResult();
        List<String> activeActivityIds = Collections.emptyList();
        String processDefinitionId;
        if (processInstance != null) {
            activeActivityIds = runtimeService.getActiveActivityIds(processInstanceId);
            processDefinitionId = processInstance.getProcessDefinitionId();
        } else {
            HistoricProcessInstance historicProcessInstance = historyService.createHistoricProcessInstanceQuery()
                .processInstanceId(processInstanceId).singleResult();
            Assert.notNull(historicProcessInstance, ActivitiConstant.Message.PROCESS_INSTANCE_NOT_EXIST);
            processDefinitionId = historicProcessInstance.getProcessDefinitionId();
        }

        BpmnModel model = repositoryService.getBpmnModel(processDefinitionId);
        if (model.getLocationMap().size() > 0) {
            ProcessDiagramGenerator generator = new DefaultProcessDiagramGenerator();
            return generator.generateDiagram(model, ActivitiConstant.IMAGE_TYPE,
                activeActivityIds,
                Collections.emptyList(), ActivitiConstant.FONT_NAME, ActivitiConstant.FONT_NAME, ActivitiConstant.FONT_NAME, null, 1.0);
        }
        return null;
    }

    private AuditResult getAuditResult(List<Task> taskList, String userId, String processInstanceId, AuditStatusEnum unAudit) {

        AuditResult result = new AuditResult();
        result.setProcessRootId(processInstanceId);
        result.setSubmitUser(userId);
        result.setAuditStatus(unAudit.getValue());
        result.setNextAuditUserList(taskList.stream().map(TaskInfo::getAssignee).collect(Collectors.toSet()));
        taskList.forEach(task -> {
            if (userId.equals(task.getAssignee())) {
                result.setNextTaskId(task.getId());
            }
        });
        return result;
    }

}
