/*
 * Decompiled with CFR 0.152.
 */
package com.sap.db.jdbc;

import com.sap.db.annotations.NotThreadSafe;
import com.sap.db.jdbc.ObjectStoreFile;
import com.sap.db.util.ByteUtils;
import com.sap.db.util.UUIDUtils;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.UUID;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

@NotThreadSafe
class ObjectStore {
    private final int ST_UNENCRYPTED_HEADER_SIZE = 64;
    private final int ST_CURRENT_FILE_VERSION = 0;
    private final int F_STORE_IS_INITIALIZED = 1;
    private final int F_STORE_IS_ENCRYPTED = 2;
    private final int ST_ENCRYPTED_HEADER_SIZE = 128;
    private final int ST_ALIGNMENT = 16;
    private final int F_STORE_NEEDS_RECOVERY = 1;
    private final int ST_INDEX_PAGE_SIZE = 4096;
    private final int ST_INDEX_ENTRY_SIZE = 128;
    private final int ST_OBJECT_NAME_MAX_SIZE = 80;
    private final int ST_INDEX_ENTRY_NAME_SIZE = 84;
    private final short T_FREE = 0;
    private final short T_INDEX_PAGE = 1;
    private final short T_OBJECT = (short)2;
    private final short ENC_ALG_NONE = 0;
    private final short ENC_ALG_AES_256_CBC = 1;
    private final int ST_HMAC_256_HASH_SIZE = 32;
    private ObjectStoreFile _file;
    private boolean _isOpen = false;
    private long _fileSize;
    private UnencryptedHeader _unencryptedHeader;
    private EncryptedHeader _encryptedHeader;
    private IndexPage[] _indexPages;
    private int _indexSize;
    private SecureRandom _randomGenerator = new SecureRandom();
    private int _lastCounter;
    private byte[] _aesKeyBytes;
    private SecretKey _aesKey;
    private boolean _recreateStore;
    static final int InvalidIndexNum = -1;
    static final int InvalidObjectHandle = -1;
    static final int SHA256_BLOCK_SIZE = 64;
    static final int SHA256_OUTPUT_SIZE = 32;

    ObjectStore() {
    }

    void open(String filename, String password) throws StException {
        this.open(filename, password, true);
    }

    void open(String filename, String password, boolean create) throws StException {
        if (this._file == null) {
            this._file = new ObjectStoreFile();
        }
        this._recreateStore = false;
        boolean openedExisting = this._file.open(filename, false, false);
        if (openedExisting) {
            this._file.lock();
            try {
                this._openExistingStore(password);
            }
            catch (StException e) {
                this.close();
                throw e;
            }
            finally {
                this._file.unlock();
            }
        }
        if (this._recreateStore) {
            this._file.close();
        }
        if (!openedExisting || this._recreateStore) {
            if (!create) {
                throw new StException(ErrorCodes.FILE_OPEN_FAILED);
            }
            if (!this._file.open(filename, true, this._recreateStore)) {
                throw new StException(ErrorCodes.FILE_OPEN_FAILED);
            }
            this._file.lock();
            try {
                this._createNewStore(password);
            }
            catch (StException e) {
                this.close();
                throw e;
            }
            finally {
                this._file.unlock();
            }
        }
    }

    private void _openExistingStore(String password) throws StException {
        this._fileSize = this._file.getFileSize();
        if (this._fileSize == 0L) {
            this._recreateStore = true;
            return;
        }
        if (this._fileSize < 64L) {
            throw new StException(ErrorCodes.INVALID_STORE);
        }
        this._loadUnencryptedHeader();
        if ((this._unencryptedHeader._flags & 1) == 0) {
            this._recreateStore = true;
            return;
        }
        if (this._unencryptedHeader._version > 0) {
            throw new StException(ErrorCodes.UNKNOWN_STORE_VERSION);
        }
        if ((this._unencryptedHeader._flags & 2) != 0 && password == null) {
            throw new StException(ErrorCodes.PASSWORD_REQUIRED);
        }
        if ((this._unencryptedHeader._flags & 2) == 0 && password != null) {
            throw new StException(ErrorCodes.UNENCRYPTED_STORE);
        }
        if (password != null) {
            this._aesKeyBytes = this._generateEncryptionKey(password);
            this._aesKey = new SecretKeySpec(this._aesKeyBytes, 0, 32, "AES");
        }
        this._loadEncryptedHeader();
        if ((this._encryptedHeader._flags & 1) != 0) {
            this._doRecovery();
        }
        this._loadIndexPages();
        this._isOpen = true;
    }

    private void _createNewStore(String password) throws StException {
        this._unencryptedHeader = new UnencryptedHeader();
        if (password != null) {
            if (!this._checkPasswordAgainstPolicy(password)) {
                throw new StException(ErrorCodes.WEAK_PASSWORD);
            }
            this._aesKeyBytes = this._generateEncryptionKey(password);
            this._aesKey = new SecretKeySpec(this._aesKeyBytes, 0, 32, "AES");
            this._unencryptedHeader._flags |= 2;
        }
        this._encryptedHeader = new EncryptedHeader();
        this._fileSize = this._align(192L, 16) + (long)this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, this._getIndexEncAlg());
        this._storeUnencryptedHeader();
        this._storeEncryptedHeader();
        this._growIndexPageArray();
        this._indexPages[0] = new IndexPage(this._encryptedHeader._indexPageSize);
        this._writeIndexPageToFile(0);
        this._unencryptedHeader._flags |= 1;
        this._storeUnencryptedHeader();
        this._file.flush();
        this._isOpen = true;
    }

    void close() {
        if (this._file != null) {
            this._file.close();
        }
        this._indexPages = null;
        this._indexSize = 0;
        this._isOpen = false;
        this._fileSize = 0L;
        this._lastCounter = 0;
        this._aesKey = null;
    }

    boolean isOpen() {
        return this._isOpen;
    }

    void addObject(String name, byte[] value) throws StException {
        this.addObject(name, value, 0, 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addObject(String name, byte[] value, int type, int flags) throws StException {
        if (name == null || name.length() == 0 || name.length() > 80 || value == null || value.length == 0 || type > 65535 || flags > 65535) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            if (this._findObjectByName(name) != -1) {
                throw new StException(ErrorCodes.OBJECT_ALREADY_EXISTS);
            }
            short encAlg = this._getDefaultEncAlg();
            int indexNum = this._allocateObject(this._getObjectSizeOnDisk(value.length, encAlg));
            IndexEntry entry = this._getIndexEntry(indexNum);
            entry._name = name;
            entry._size = value.length;
            entry._type = (short)2;
            entry._flags = 0;
            entry._user_type = (short)type;
            entry._user_flags = (short)flags;
            entry._enc_alg = encAlg;
            this._generateRandomIV(entry._iv);
            this._writeObjectToFile(entry._offset, value, entry._size_on_disk, encAlg, entry._iv);
            this._writeIndexPageToFileWithRecovery(this._getIndexPageNumFromIndexNum(indexNum));
        }
        finally {
            this._file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeObject(String name) throws StException {
        if (name == null) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            int indexNum = this._findObjectByName(name);
            if (indexNum == -1) {
                throw new StException(ErrorCodes.OBJECT_NOT_FOUND);
            }
            IndexEntry entry = this._getIndexEntry(indexNum);
            if (indexNum == this._indexSize - 1) {
                --this._indexSize;
                this._fileSize = entry._offset;
                entry.clear();
            } else {
                long tempOffset = entry._offset;
                int tempSizeOnDisk = entry._size_on_disk;
                entry.clear();
                entry._offset = tempOffset;
                entry._size_on_disk = tempSizeOnDisk;
            }
            this._writeIndexPageToFileWithRecovery(this._getIndexPageNumFromIndexNum(indexNum));
        }
        finally {
            this._file.unlock();
        }
    }

    boolean objectExists(String name) throws StException {
        if (name == null) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            boolean bl = this._findObjectByName(name) != -1;
            return bl;
        }
        finally {
            this._file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    byte[] getObject(String name) throws StException {
        if (name == null) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            int indexNum = this._findObjectByName(name);
            if (indexNum == -1) {
                throw new StException(ErrorCodes.OBJECT_NOT_FOUND);
            }
            byte[] byArray = this._getObjectByIndexNum(indexNum);
            return byArray;
        }
        finally {
            this._file.unlock();
        }
    }

    byte[] getObject(int handle) throws StException {
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        if (handle == -1) {
            throw new StException(ErrorCodes.INVALID_INDEX);
        }
        this._file.lock();
        try {
            this._refreshStore();
            if (this._getIndexEntry((int)handle)._type == 0) {
                throw new StException(ErrorCodes.OBJECT_NOT_FOUND);
            }
            byte[] byArray = this._getObjectByIndexNum(handle);
            return byArray;
        }
        finally {
            this._file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getObjectType(String name) throws StException {
        if (name == null) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            int indexNum = this._findObjectByName(name);
            if (indexNum == -1) {
                throw new StException(ErrorCodes.OBJECT_NOT_FOUND);
            }
            IndexEntry entry = this._getIndexEntry(indexNum);
            short s = entry._user_type;
            return s;
        }
        finally {
            this._file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int getObjectFlags(String name) throws StException {
        if (name == null) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            int indexNum = this._findObjectByName(name);
            if (indexNum == -1) {
                throw new StException(ErrorCodes.OBJECT_NOT_FOUND);
            }
            IndexEntry entry = this._getIndexEntry(indexNum);
            short s = entry._user_flags;
            return s;
        }
        finally {
            this._file.unlock();
        }
    }

    int findNextObject(int prev) throws StException {
        return this.findNextObject(prev, 0, 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int findNextObject(int prev, int type, int flags) throws StException {
        if (type > 65535 || flags > 65535) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        if (prev < -1) {
            return -1;
        }
        this._file.lock();
        try {
            IndexEntry entry;
            this._refreshStore();
            int indexNum = prev;
            do {
                if (++indexNum >= this._indexSize) {
                    int n = -1;
                    return n;
                }
                entry = this._getIndexEntry(indexNum);
            } while (entry._type != 2 || type != 0 && entry._user_type != type || flags != 0 && (entry._user_flags & flags) == 0);
            int n = indexNum;
            return n;
        }
        finally {
            this._file.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    String getObjectName(int handle) throws StException {
        if (handle == -1) {
            throw new StException(ErrorCodes.INVALID_PARAMETER);
        }
        if (!this._isOpen) {
            throw new StException(ErrorCodes.STORE_NOT_OPEN);
        }
        this._file.lock();
        try {
            this._refreshStore();
            if (!this._isValidIndex(handle)) {
                throw new StException(ErrorCodes.INVALID_INDEX);
            }
            IndexEntry entry = this._getIndexEntry(handle);
            if (entry._type != 2) {
                throw new StException(ErrorCodes.INVALID_INDEX);
            }
            String string = entry._name;
            return string;
        }
        finally {
            this._file.unlock();
        }
    }

    static String getUserProfilePath(boolean create) throws SQLException {
        return ObjectStoreFile.getUserProfilePath(create);
    }

    static String getUserProfilePath() throws SQLException {
        return ObjectStoreFile.getUserProfilePath(true);
    }

    void setObjectStoreFile(ObjectStoreFile file) {
        this._file = file;
    }

    private void _refreshStore() throws StException {
        this._loadEncryptedHeader();
        if (this._encryptedHeader._counter != this._lastCounter) {
            if ((this._encryptedHeader._flags & 1) != 0) {
                this._doRecovery();
            }
            this._loadIndexPages();
        }
    }

    private void _doRecovery() throws StException {
        if (this._encryptedHeader._recoverySize == 0 || this._encryptedHeader._recoverySrcOffset == 0L || this._encryptedHeader._recoveryDstOffset == 0L) {
            throw new StException(ErrorCodes.INVALID_STORE);
        }
        byte[] buf = new byte[this._encryptedHeader._recoverySize];
        this._rawReadFromFile(this._encryptedHeader._recoverySrcOffset, buf);
        this._rawWriteToFile(this._encryptedHeader._recoveryDstOffset, buf);
        if ((this._encryptedHeader._flags & 1) != 0) {
            this._encryptedHeader._flags ^= 1;
        }
        this._storeEncryptedHeader();
    }

    private boolean _isEncrypted() {
        return this._aesKey != null;
    }

    private short _getDefaultEncAlg() {
        if (this._isEncrypted()) {
            return 1;
        }
        return 0;
    }

    private short _getIndexEncAlg() {
        if (this._isEncrypted()) {
            return 1;
        }
        return 0;
    }

    private void _loadUnencryptedHeader() throws StException {
        byte[] buf = new byte[64];
        this._rawReadFromFile(0L, buf);
        this._unencryptedHeader = new UnencryptedHeader(buf);
        if (!this._unencryptedHeader._fixedString.equals("HDBObjectStore")) {
            throw new StException(ErrorCodes.INVALID_STORE);
        }
    }

    private void _storeUnencryptedHeader() throws StException {
        this._rawWriteToFile(0L, this._unencryptedHeader.getData());
    }

    private void _loadEncryptedHeader() throws StException {
        byte[] buf = new byte[128];
        this._rawReadFromFile(this._unencryptedHeader._size, buf);
        if (this._isEncrypted()) {
            this._doDecrypt(buf, 0, 128, buf, 0, 128, this._unencryptedHeader._uuid, null);
        }
        this._encryptedHeader = new EncryptedHeader(buf);
        if (!this._encryptedHeader._fixedString.equals("HDBObjectStore")) {
            throw new StException(ErrorCodes.INVALID_PASSWORD);
        }
    }

    private void _storeEncryptedHeader() throws StException {
        if (this._isEncrypted()) {
            byte[] buf = new byte[128];
            byte[] header = this._encryptedHeader.getData();
            this._doEncrypt(header, 0, header.length, buf, 0, 128, this._unencryptedHeader._uuid, null);
            this._rawWriteToFile(this._unencryptedHeader._size, buf);
        } else {
            this._rawWriteToFile(this._unencryptedHeader._size, this._encryptedHeader.getData());
        }
    }

    private void _loadIndexPages() throws StException {
        IndexEntry entry;
        long fileOffset = this._align(this._unencryptedHeader._size + this._encryptedHeader._size);
        int indexPageNum = 0;
        byte[] iv = this._encryptedHeader._firstPageIV;
        short encAlg = this._getIndexEncAlg();
        while (true) {
            if (this._indexPages == null || indexPageNum == this._indexPages.length) {
                this._growIndexPageArray();
            }
            byte[] buf = new byte[this._encryptedHeader._indexPageSize];
            this._readObjectFromFile(fileOffset, buf, this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, encAlg), encAlg, iv);
            this._indexPages[indexPageNum] = new IndexPage(buf);
            entry = this._getIndexEntryOnPage(indexPageNum, this._indexEntriesPerPage() - 1);
            if (entry._type == 0) break;
            if (entry._type != 1 || entry._size != this._encryptedHeader._indexPageSize) {
                throw new StException(ErrorCodes.INVALID_STORE);
            }
            fileOffset = entry._offset;
            encAlg = entry._enc_alg;
            iv = entry._iv;
            ++indexPageNum;
        }
        int i = this._indexEntriesPerPage() - 1;
        this._indexSize = indexPageNum * this._indexEntriesPerPage();
        while (true) {
            entry = this._getIndexEntryOnPage(indexPageNum, i);
            if (entry._type != 0) {
                this._indexSize += i + 1;
                this._fileSize = entry._offset + (long)entry._size_on_disk;
                break;
            }
            if (i == 0) {
                this._fileSize = this._getFileOffsetForIndexPage(indexPageNum) + (long)this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, this._getIndexEncAlg());
                break;
            }
            --i;
        }
        this._lastCounter = this._encryptedHeader._counter;
    }

    private int _getObjectSizeOnDisk(int objectSize, short encType) {
        switch (encType) {
            case 0: {
                return this._align(objectSize);
            }
            case 1: {
                return this._align(objectSize + 32);
            }
        }
        return objectSize;
    }

    private int _findObjectByName(String name) throws StException {
        for (int indexNum = 0; indexNum < this._indexSize; ++indexNum) {
            IndexEntry entry = this._getIndexEntry(indexNum);
            if (entry._name == null || !entry._name.equalsIgnoreCase(name)) continue;
            return indexNum;
        }
        return -1;
    }

    private byte[] _getObjectByIndexNum(int indexNum) throws StException {
        if (!this._isValidIndex(indexNum)) {
            throw new StException(ErrorCodes.INVALID_INDEX);
        }
        IndexEntry entry = this._getIndexEntry(indexNum);
        if (entry._type != 2) {
            throw new StException(ErrorCodes.INVALID_INDEX);
        }
        if (entry._offset == 0L) {
            throw new StException(ErrorCodes.INVALID_STORE);
        }
        byte[] buf = new byte[entry._size];
        this._readObjectFromFile(entry._offset, buf, entry._size_on_disk, entry._enc_alg, entry._iv);
        return buf;
    }

    private void _generateRandomIV(byte[] buf) {
        if (this._isEncrypted()) {
            this._randomGenerator.nextBytes(buf);
        }
    }

    private byte[] _generateEncryptionKey(String password) {
        byte[] key = new byte[32];
        ObjectStore.doPbkdf2HmacSha256(key, password, this._unencryptedHeader._uuid, 25000);
        return key;
    }

    private boolean _checkPasswordAgainstPolicy(String password) {
        boolean hasUpper = false;
        boolean hasLower = false;
        boolean hasNumber = false;
        int passwordLength = password.length();
        if (passwordLength < 8) {
            return false;
        }
        for (int i = 0; i < passwordLength; ++i) {
            char ch = password.charAt(i);
            if (Character.isUpperCase(ch)) {
                hasUpper = true;
                continue;
            }
            if (Character.isLowerCase(ch)) {
                hasLower = true;
                continue;
            }
            if (!Character.isDigit(ch)) continue;
            hasNumber = true;
        }
        return hasUpper && hasLower && hasNumber;
    }

    private int _allocateObject(int sizeOnDisk) throws StException {
        IndexEntry entry;
        int indexNum;
        for (indexNum = 0; indexNum < this._indexSize; ++indexNum) {
            entry = this._getIndexEntry(indexNum);
            if (entry._type != 0 || entry._size_on_disk < sizeOnDisk) continue;
            if (entry._offset == 0L) {
                throw new StException(ErrorCodes.INVALID_STORE);
            }
            return indexNum;
        }
        if ((this._indexSize + 1) % this._indexEntriesPerPage() == 0) {
            this._allocateNewIndexPage();
        }
        indexNum = this._indexSize++;
        entry = this._getIndexEntry(indexNum);
        entry._offset = this._fileSize;
        entry._size_on_disk = sizeOnDisk;
        this._fileSize += (long)sizeOnDisk;
        return indexNum;
    }

    private void _growIndexPageArray() {
        if (this._indexPages == null) {
            this._indexPages = new IndexPage[32];
        } else {
            IndexPage[] temp = new IndexPage[this._indexPages.length * 2];
            System.arraycopy(this._indexPages, 0, temp, 0, this._indexPages.length);
            this._indexPages = temp;
        }
    }

    private void _allocateNewIndexPage() throws StException {
        long offset = this._fileSize;
        int newPageNum = (this._indexSize + 1) / this._indexEntriesPerPage();
        if (newPageNum == this._indexPages.length) {
            this._growIndexPageArray();
        }
        this._indexPages[newPageNum] = new IndexPage(this._encryptedHeader._indexPageSize);
        short encAlg = this._getIndexEncAlg();
        IndexEntry entry = this._getIndexEntryForIndexPage(newPageNum);
        entry._name = "__indexpage" + newPageNum;
        entry._size = this._encryptedHeader._indexPageSize;
        entry._offset = offset;
        entry._size_on_disk = this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, encAlg);
        entry._type = 1;
        entry._enc_alg = encAlg;
        this._fileSize += (long)this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, encAlg);
        this._writeIndexPageToFile(newPageNum);
        this._writeIndexPageToFileWithRecovery(newPageNum - 1);
        ++this._indexSize;
    }

    private IndexEntry _getIndexEntryForIndexPage(int pageNum) {
        if (pageNum == 0) {
            return null;
        }
        return this._indexPages[pageNum - 1]._entries[this._indexEntriesPerPage() - 1];
    }

    private long _getFileOffsetForIndexPage(int pageNum) {
        if (pageNum == 0) {
            return this._align(this._unencryptedHeader._size + this._encryptedHeader._size);
        }
        return this._getIndexEntryForIndexPage((int)pageNum)._offset;
    }

    private void _writeIndexPageToFileWithRecovery(int pageNum) throws StException {
        long offset = this._getFileOffsetForIndexPage(pageNum);
        int sizeOnDisk = this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, this._getIndexEncAlg());
        byte[] buf = new byte[sizeOnDisk];
        this._rawReadFromFile(offset, buf);
        this._fileSize += (long)sizeOnDisk;
        this._rawWriteToFile(this._fileSize - (long)sizeOnDisk, buf);
        this._file.flush();
        this._encryptedHeader._flags |= 1;
        this._encryptedHeader._recoverySrcOffset = this._fileSize - (long)sizeOnDisk;
        this._encryptedHeader._recoveryDstOffset = offset;
        this._encryptedHeader._recoverySize = sizeOnDisk;
        this._updateWriteCounter();
        this._storeEncryptedHeader();
        this._file.flush();
        this._writeIndexPageToFile(pageNum);
        this._file.flush();
        this._encryptedHeader._flags ^= 1;
        this._fileSize -= (long)sizeOnDisk;
        this._storeEncryptedHeader();
        this._file.flush();
    }

    private void _writeIndexPageToFile(int pageNum) throws StException {
        long offset = this._getFileOffsetForIndexPage(pageNum);
        byte[] iv = pageNum == 0 ? this._encryptedHeader._firstPageIV : this._getIndexEntryForIndexPage((int)pageNum)._iv;
        this._writeObjectToFile(offset, this._indexPages[pageNum].getBytes(), this._getObjectSizeOnDisk(this._encryptedHeader._indexPageSize, this._getIndexEncAlg()), this._getIndexEncAlg(), iv);
    }

    private void _updateWriteCounter() throws StException {
        ++this._lastCounter;
        ++this._encryptedHeader._counter;
    }

    static void doHmacSha256(byte[] key, byte[] msg, int msgOffset, int msgLen, byte[] hmac) {
        byte[] opad = new byte[64];
        byte[] ipad = new byte[64];
        byte[] k = new byte[64];
        try {
            if (key.length > 64) {
                MessageDigest digest = MessageDigest.getInstance("SHA-256");
                digest.update(key);
                byte[] h = digest.digest();
                System.arraycopy(h, 0, k, 0, 32);
            } else {
                System.arraycopy(key, 0, k, 0, key.length);
            }
            for (int i = 0; i < 64; ++i) {
                opad[i] = (byte)(0x5C ^ k[i]);
                ipad[i] = (byte)(0x36 ^ k[i]);
            }
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            digest.update(ipad);
            digest.update(msg, msgOffset, msgLen);
            byte[] h1 = digest.digest();
            digest = MessageDigest.getInstance("SHA-256");
            digest.update(opad);
            digest.update(h1);
            byte[] h2 = digest.digest();
            System.arraycopy(h2, 0, hmac, 0, 32);
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            // empty catch block
        }
    }

    static void doPbkdf2HmacSha256(byte[] DK, String password, byte[] salt, int c) {
        int i = 1;
        int MAX_SALT_LEN = 64;
        byte[] init = new byte[68];
        byte[] T = new byte[32];
        byte[] U = new byte[32];
        byte[] pwd = password.getBytes();
        int DKlen = DK.length;
        int DKpos = 0;
        int saltLen = salt.length;
        System.arraycopy(salt, 0, init, 0, saltLen);
        while (DKlen > 0) {
            init[saltLen] = (byte)(i >> 24 & 0xFF);
            init[saltLen + 1] = (byte)(i >> 16 & 0xFF);
            init[saltLen + 2] = (byte)(i >> 8 & 0xFF);
            init[saltLen + 3] = (byte)(i & 0xFF);
            ObjectStore.doHmacSha256(pwd, init, 0, saltLen + 4, U);
            System.arraycopy(U, 0, T, 0, 32);
            for (int j = 2; j <= c; ++j) {
                ObjectStore.doHmacSha256(pwd, U, 0, 32, U);
                for (int k = 0; k < 32; ++k) {
                    int n = k;
                    T[n] = (byte)(T[n] ^ U[k]);
                }
            }
            int amt = DKlen < 32 ? DKlen : 32;
            System.arraycopy(T, 0, DK, DKpos, amt);
            DKpos += amt;
            DKlen -= amt;
            ++i;
        }
    }

    private void _doEncrypt(byte[] src, int srcOffset, int srcLen, byte[] dst, int dstOffset, int dstLen, byte[] iv, byte[] hmac) throws StException {
        try {
            int writtenLen;
            byte[] fill = new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
            Cipher cipher = Cipher.getInstance("AES/CBC/NoPadding");
            cipher.init(1, (Key)this._aesKey, new IvParameterSpec(iv));
            dstOffset += cipher.update(src, srcOffset, srcLen, dst, dstOffset);
            for (int fillLen = dstLen - srcLen; fillLen > 0; fillLen -= writtenLen) {
                int curLen = fillLen;
                if (curLen > 16) {
                    curLen = 16;
                }
                writtenLen = cipher.update(fill, 0, curLen, dst, dstOffset);
                dstOffset += writtenLen;
            }
            cipher.doFinal(dst, dstOffset);
            if (hmac != null) {
                ObjectStore.doHmacSha256(this._aesKeyBytes, src, srcOffset, srcLen, hmac);
            }
        }
        catch (InvalidKeyException e) {
            throw new StException(ErrorCodes.CRYPTO_EXTENSION_NOT_INSTALLED);
        }
        catch (Exception e) {
            throw new StException(ErrorCodes.ENCRYPT_FAILED);
        }
    }

    private void _doDecrypt(byte[] src, int srcOffset, int srcLen, byte[] dst, int dstOffset, int dstLen, byte[] iv, byte[] hmac) throws StException {
        try {
            Cipher cipher = Cipher.getInstance("AES/CBC/NoPadding");
            cipher.init(2, (Key)this._aesKey, new IvParameterSpec(iv));
            cipher.doFinal(src, srcOffset, srcLen, src, srcOffset);
            if (src != dst) {
                System.arraycopy(src, srcOffset, dst, dstOffset, dstLen);
            }
            if (hmac != null) {
                byte[] hmac2 = new byte[32];
                ObjectStore.doHmacSha256(this._aesKeyBytes, dst, dstOffset, dstLen, hmac2);
                if (!Arrays.equals(hmac, hmac2)) {
                    throw new StException(ErrorCodes.HMAC_CHECK_FAILED);
                }
            }
        }
        catch (InvalidKeyException e) {
            throw new StException(ErrorCodes.CRYPTO_EXTENSION_NOT_INSTALLED);
        }
        catch (Exception e) {
            throw new StException(ErrorCodes.DECRYPT_FAILED);
        }
    }

    private void _writeObjectToFile(long offset, byte[] object, int sizeOnDisk, short encType, byte[] iv) throws StException {
        if (encType == 1) {
            byte[] buf = new byte[sizeOnDisk];
            byte[] hmac = new byte[32];
            this._doEncrypt(object, 0, object.length, buf, 32, sizeOnDisk - 32, iv, hmac);
            System.arraycopy(hmac, 0, buf, 0, 32);
            this._rawWriteToFile(offset, buf);
        } else {
            this._rawWriteToFile(offset, object);
        }
    }

    private void _readObjectFromFile(long offset, byte[] object, int sizeOnDisk, short encType, byte[] iv) throws StException {
        if (encType == 1) {
            byte[] buf = new byte[sizeOnDisk];
            byte[] hmac = new byte[32];
            this._rawReadFromFile(offset, buf);
            System.arraycopy(buf, 0, hmac, 0, 32);
            this._doDecrypt(buf, 32, sizeOnDisk - 32, object, 0, object.length, iv, hmac);
        } else {
            this._rawReadFromFile(offset, object);
        }
    }

    private void _rawWriteToFile(long offset, byte[] buf) throws StException {
        if (!this._file.position(offset)) {
            throw new StException(ErrorCodes.FILE_WRITE_FAILED);
        }
        if (!this._file.write(buf)) {
            throw new StException(ErrorCodes.FILE_WRITE_FAILED);
        }
    }

    private void _rawReadFromFile(long offset, byte[] buf) throws StException {
        if (!this._file.position(offset)) {
            throw new StException(ErrorCodes.FILE_WRITE_FAILED);
        }
        if (!this._file.read(buf)) {
            throw new StException(ErrorCodes.FILE_WRITE_FAILED);
        }
    }

    private int _indexEntriesPerPage() {
        return this._encryptedHeader._indexPageSize / 128;
    }

    private int _getIndexPageNumFromIndexNum(int indexNum) {
        return indexNum / this._indexEntriesPerPage();
    }

    private boolean _isValidIndex(int pageNum, int indexNumOnPage) {
        return pageNum * this._indexEntriesPerPage() + indexNumOnPage < this._indexSize;
    }

    private boolean _isValidIndex(int indexNum) {
        return indexNum < this._indexSize;
    }

    private IndexEntry _getIndexEntryOnPage(int pageNum, int indexNumOnPage) {
        return this._indexPages[pageNum]._entries[indexNumOnPage];
    }

    private IndexEntry _getIndexEntry(int indexNum) {
        return this._getIndexEntryOnPage(indexNum / this._indexEntriesPerPage(), indexNum % this._indexEntriesPerPage());
    }

    private long _align(long x, int alignment) {
        return (x + (long)alignment - 1L) / (long)alignment * (long)alignment;
    }

    private long _align(long x) {
        return this._align(x, this._encryptedHeader._alignment);
    }

    private int _align(int x, int alignment) {
        return (int)this._align((long)x, alignment);
    }

    private int _align(int x) {
        return this._align(x, this._encryptedHeader._alignment);
    }

    private boolean _isAligned(long x) {
        return x == this._align(x);
    }

    private class IndexPage {
        IndexEntry[] _entries;
        int _pageSize;

        IndexPage(int pageSize) {
            this.init(pageSize);
            for (int i = 0; i < this._entries.length; ++i) {
                this._entries[i] = new IndexEntry();
            }
        }

        IndexPage(byte[] buf) {
            this.init(buf.length);
            for (int i = 0; i < this._entries.length; ++i) {
                this._entries[i] = new IndexEntry(buf, i * 128);
            }
        }

        private void init(int pageSize) {
            this._pageSize = pageSize;
            this._entries = new IndexEntry[pageSize / 128];
        }

        byte[] getBytes() {
            byte[] buf = new byte[this._pageSize];
            for (int i = 0; i < this._entries.length; ++i) {
                this._entries[i].getBytes(buf, i * 128);
            }
            return buf;
        }
    }

    private class IndexEntry {
        String _name;
        int _size;
        long _offset;
        int _size_on_disk;
        short _type;
        short _flags;
        short _user_type;
        short _user_flags;
        short _enc_alg;
        byte[] _iv = new byte[16];

        IndexEntry() {
            this._type = 0;
        }

        IndexEntry(byte[] buf, int offset) {
            this._name = new String(buf, offset, 84).trim();
            this._size = ByteUtils.getInt(buf, offset + 84);
            this._offset = ByteUtils.getLong(buf, offset + 88);
            this._size_on_disk = ByteUtils.getInt(buf, offset + 96);
            this._type = ByteUtils.getShort(buf, offset + 100);
            this._flags = ByteUtils.getShort(buf, offset + 102);
            this._user_type = ByteUtils.getShort(buf, offset + 104);
            this._user_flags = ByteUtils.getShort(buf, offset + 106);
            this._enc_alg = ByteUtils.getShort(buf, offset + 108);
            System.arraycopy(buf, offset + 110, this._iv, 0, 16);
        }

        void clear() {
            this._name = null;
            this._size = 0;
            this._offset = 0L;
            this._size_on_disk = 0;
            this._type = 0;
            this._flags = 0;
            this._user_type = 0;
            this._user_flags = 0;
            this._enc_alg = 0;
            this._iv = new byte[16];
        }

        void getBytes(byte[] buf, int offset) {
            if (this._name != null) {
                System.arraycopy(this._name.getBytes(), 0, buf, offset, this._name.length());
            }
            ByteUtils.putInt(this._size, buf, offset + 84);
            ByteUtils.putLong(this._offset, buf, offset + 88);
            ByteUtils.putInt(this._size_on_disk, buf, offset + 96);
            ByteUtils.putShort(this._type, buf, offset + 100);
            ByteUtils.putShort(this._flags, buf, offset + 102);
            ByteUtils.putShort(this._user_type, buf, offset + 104);
            ByteUtils.putShort(this._user_flags, buf, offset + 106);
            ByteUtils.putShort(this._enc_alg, buf, offset + 108);
            System.arraycopy(this._iv, 0, buf, offset + 110, 16);
        }
    }

    private class EncryptedHeader {
        String _fixedString;
        int _flags;
        int _size;
        int _alignment;
        int _counter;
        byte[] _firstPageIV = new byte[16];
        int _indexPageSize;
        int _recoverySize;
        long _recoverySrcOffset;
        long _recoveryDstOffset;

        EncryptedHeader() {
            this._fixedString = "HDBObjectStore";
            this._flags = 0;
            this._size = 128;
            this._alignment = 16;
            this._counter = 0;
            ObjectStore.this._generateRandomIV(this._firstPageIV);
            this._indexPageSize = 4096;
            this._recoverySize = 0;
            this._recoverySrcOffset = 0L;
            this._recoveryDstOffset = 0L;
        }

        EncryptedHeader(byte[] buf) {
            this._fixedString = new String(buf, 0, 16).trim();
            this._flags = ByteUtils.getInt(buf, 16);
            this._size = ByteUtils.getInt(buf, 20);
            this._alignment = ByteUtils.getInt(buf, 24);
            this._counter = ByteUtils.getInt(buf, 28);
            System.arraycopy(buf, 32, this._firstPageIV, 0, 16);
            this._indexPageSize = ByteUtils.getInt(buf, 48);
            this._recoverySize = ByteUtils.getInt(buf, 52);
            this._recoverySrcOffset = ByteUtils.getLong(buf, 56);
            this._recoveryDstOffset = ByteUtils.getLong(buf, 64);
        }

        byte[] getData() {
            byte[] buf = new byte[128];
            System.arraycopy(this._fixedString.getBytes(), 0, buf, 0, this._fixedString.length());
            ByteUtils.putInt(this._flags, buf, 16);
            ByteUtils.putInt(this._size, buf, 20);
            ByteUtils.putInt(this._alignment, buf, 24);
            ByteUtils.putInt(this._counter, buf, 28);
            System.arraycopy(this._firstPageIV, 0, buf, 32, 16);
            ByteUtils.putInt(this._indexPageSize, buf, 48);
            ByteUtils.putInt(this._recoverySize, buf, 52);
            ByteUtils.putLong(this._recoverySrcOffset, buf, 56);
            ByteUtils.putLong(this._recoveryDstOffset, buf, 64);
            return buf;
        }
    }

    private class UnencryptedHeader {
        String _fixedString;
        int _version;
        int _flags;
        int _size;
        byte[] _uuid = new byte[16];

        UnencryptedHeader() {
            this._fixedString = "HDBObjectStore";
            this._version = 0;
            this._flags = 0;
            this._size = 64;
            System.arraycopy(UUIDUtils.getBytes(UUID.randomUUID()), 0, this._uuid, 0, 16);
        }

        UnencryptedHeader(byte[] buf) {
            this._fixedString = new String(buf, 0, 16).trim();
            this._version = ByteUtils.getInt(buf, 16);
            this._flags = ByteUtils.getInt(buf, 20);
            this._size = ByteUtils.getInt(buf, 24);
            System.arraycopy(buf, 32, this._uuid, 0, 16);
        }

        byte[] getData() {
            byte[] buf = new byte[64];
            System.arraycopy(this._fixedString.getBytes(), 0, buf, 0, this._fixedString.length());
            ByteUtils.putInt(this._version, buf, 16);
            ByteUtils.putInt(this._flags, buf, 20);
            ByteUtils.putInt(this._size, buf, 24);
            System.arraycopy(this._uuid, 0, buf, 32, 16);
            return buf;
        }
    }

    static class StException
    extends Exception {
        private ErrorCodes _code;

        StException(ErrorCodes code) {
            this._code = code;
        }

        ErrorCodes getCode() {
            return this._code;
        }
    }

    static enum ErrorCodes {
        INVALID_PARAMETER,
        STORE_NOT_OPEN,
        OBJECT_NOT_FOUND,
        OBJECT_ALREADY_EXISTS,
        FILE_READ_FAILED,
        FILE_WRITE_FAILED,
        FILE_OPEN_FAILED,
        INVALID_STORE,
        INVALID_PASSWORD,
        PASSWORD_REQUIRED,
        UNENCRYPTED_STORE,
        INVALID_INDEX,
        HMAC_CHECK_FAILED,
        ENCRYPT_FAILED,
        DECRYPT_FAILED,
        CRYPTO_EXTENSION_NOT_INSTALLED,
        UNKNOWN_STORE_VERSION,
        WEAK_PASSWORD;

    }
}

