/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.cloudplatform.servlet.response.serializer;

import java.lang.reflect.Type;
import java.time.LocalDate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.sap.cloud.sdk.cloudplatform.exception.StringParsingException;
import com.sap.cloud.sdk.cloudplatform.util.StringConverter;

/**
 * Class for serialization of {@link LocalDate} from and to {@link JsonElement}.
 */
public class LocalDateGsonSerializer implements GsonSerializer<LocalDate>
{
    @Override
    @Nonnull
    public JsonElement serialize(
        @Nullable final LocalDate src,
        @Nonnull final Type typeOfSrc,
        @Nonnull final JsonSerializationContext context )
    {
        final String str = StringConverter.toString(src);

        if( str == null ) {
            return JsonNull.INSTANCE;
        }

        return new JsonPrimitive(str);
    }

    @Override
    @Nullable
    public LocalDate deserialize(
        @Nonnull final JsonElement json,
        @Nonnull final Type typeOfT,
        @Nonnull final JsonDeserializationContext context )
        throws JsonParseException
    {
        try {
            return StringConverter.toLocalDate(json.toString().replace("\"", ""));
        }
        catch( final StringParsingException e ) {
            throw new JsonParseException(e);
        }
    }
}
