/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.log;

import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import java.io.IOException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.accumulo.core.data.Mutation;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.metadata.RootTable;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.tserver.log.MutationReceiver;
import org.apache.accumulo.tserver.log.RecoveryLogsIterator;
import org.apache.accumulo.tserver.logger.LogEvents;
import org.apache.accumulo.tserver.logger.LogFileKey;
import org.apache.accumulo.tserver.logger.LogFileValue;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortedLogRecovery {
    private static final Logger log = LoggerFactory.getLogger(SortedLogRecovery.class);
    private VolumeManager fs;

    public SortedLogRecovery(VolumeManager fs) {
        this.fs = fs;
    }

    static LogFileKey maxKey(LogEvents event) {
        LogFileKey key = new LogFileKey();
        key.event = event;
        key.tabletId = Integer.MAX_VALUE;
        key.seq = Long.MAX_VALUE;
        return key;
    }

    static LogFileKey maxKey(LogEvents event, int tabletId) {
        LogFileKey key = SortedLogRecovery.maxKey(event);
        key.tabletId = tabletId;
        return key;
    }

    static LogFileKey minKey(LogEvents event) {
        LogFileKey key = new LogFileKey();
        key.event = event;
        key.tabletId = -1;
        key.seq = 0L;
        return key;
    }

    static LogFileKey minKey(LogEvents event, int tabletId) {
        LogFileKey key = SortedLogRecovery.minKey(event);
        key.tabletId = tabletId;
        return key;
    }

    private int findMaxTabletId(KeyExtent extent, List<Path> recoveryLogs) throws IOException {
        int tabletId = -1;
        try (RecoveryLogsIterator rli = new RecoveryLogsIterator(this.fs, recoveryLogs, SortedLogRecovery.minKey(LogEvents.DEFINE_TABLET), SortedLogRecovery.maxKey(LogEvents.DEFINE_TABLET));){
            KeyExtent alternative = extent;
            if (extent.isRootTablet()) {
                alternative = RootTable.OLD_EXTENT;
            }
            while (rli.hasNext()) {
                LogFileKey key = (LogFileKey)rli.next().getKey();
                Preconditions.checkState((key.event == LogEvents.DEFINE_TABLET ? 1 : 0) != 0);
                if (!key.tablet.equals((Object)extent) && !key.tablet.equals((Object)alternative)) continue;
                Preconditions.checkState((key.tabletId >= 0 ? 1 : 0) != 0, (String)"tabletId %s for %s is negative", (int)key.tabletId, (Object)extent);
                Preconditions.checkState((tabletId == -1 || key.tabletId >= tabletId ? 1 : 0) != 0);
                if (tabletId == key.tabletId) continue;
                tabletId = key.tabletId;
            }
        }
        return tabletId;
    }

    private Map.Entry<Integer, List<Path>> findLogsThatDefineTablet(KeyExtent extent, List<Path> recoveryLogs) throws IOException {
        HashMap<Integer, ArrayList<Path>> logsThatDefineTablet = new HashMap<Integer, ArrayList<Path>>();
        for (Path wal : recoveryLogs) {
            int tabletId = this.findMaxTabletId(extent, Collections.singletonList(wal));
            if (tabletId != -1) {
                ArrayList<Path> perIdList = (ArrayList<Path>)logsThatDefineTablet.get(tabletId);
                if (perIdList == null) {
                    perIdList = new ArrayList<Path>();
                    logsThatDefineTablet.put(tabletId, perIdList);
                }
                perIdList.add(wal);
                log.debug("Found tablet {} with id {} in recovery log {}", new Object[]{extent, tabletId, wal.getName()});
                continue;
            }
            log.debug("Did not find tablet {} in recovery log {}", (Object)extent, (Object)wal.getName());
        }
        if (logsThatDefineTablet.isEmpty()) {
            return new AbstractMap.SimpleEntry<Integer, List<Path>>(-1, Collections.emptyList());
        }
        return Collections.max(logsThatDefineTablet.entrySet(), (o1, o2) -> Integer.compare((Integer)o1.getKey(), (Integer)o2.getKey()));
    }

    private String getPathSuffix(String pathString) {
        Path path = new Path(pathString);
        if (path.depth() < 2) {
            throw new IllegalArgumentException("Bad path " + pathString);
        }
        return path.getParent().getName() + "/" + path.getName();
    }

    private long findRecoverySeq(List<Path> recoveryLogs, Set<String> tabletFiles, int tabletId) throws IOException {
        HashSet<String> suffixes = new HashSet<String>();
        for (String path : tabletFiles) {
            suffixes.add(this.getPathSuffix(path));
        }
        long lastStart = 0L;
        long lastFinish = 0L;
        long recoverySeq = 0L;
        try (RecoveryLogsIterator rli = new RecoveryLogsIterator(this.fs, recoveryLogs, SortedLogRecovery.minKey(LogEvents.COMPACTION_START, tabletId), SortedLogRecovery.maxKey(LogEvents.COMPACTION_START, tabletId));){
            DeduplicatingIterator ddi = new DeduplicatingIterator(rli);
            String lastStartFile = null;
            LogEvents lastEvent = null;
            while (ddi.hasNext()) {
                LogFileKey key = (LogFileKey)ddi.next().getKey();
                Preconditions.checkState((key.seq >= 0L ? 1 : 0) != 0, (String)"Unexpected negative seq %s for tabletId %s", (long)key.seq, (int)tabletId);
                Preconditions.checkState((key.tabletId == tabletId ? 1 : 0) != 0);
                Preconditions.checkState((key.seq >= Math.max(lastFinish, lastStart) ? 1 : 0) != 0);
                switch (key.event) {
                    case COMPACTION_START: {
                        lastStart = key.seq;
                        lastStartFile = key.filename;
                        break;
                    }
                    case COMPACTION_FINISH: {
                        Preconditions.checkState((key.seq > lastStart ? 1 : 0) != 0, (String)"Compaction finish <= start %s %s %s", (Object)key.tabletId, (Object)key.seq, (Object)lastStart);
                        Preconditions.checkState((lastEvent != LogEvents.COMPACTION_FINISH ? 1 : 0) != 0, (String)"Saw consecutive COMPACTION_FINISH events %s %s %s", (Object)key.tabletId, (Object)lastFinish, (Object)key.seq);
                        lastFinish = key.seq;
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Non compaction event seen " + (Object)((Object)key.event));
                    }
                }
                lastEvent = key.event;
            }
            if (lastEvent == LogEvents.COMPACTION_START && suffixes.contains(this.getPathSuffix(lastStartFile))) {
                log.debug("Considering compaction start {} {} finished because file {} in metadata table", new Object[]{tabletId, lastStart, this.getPathSuffix(lastStartFile)});
                recoverySeq = lastStart;
            } else {
                recoverySeq = Math.max(0L, lastFinish - 1L);
            }
        }
        return recoverySeq;
    }

    private void playbackMutations(List<Path> recoveryLogs, MutationReceiver mr, int tabletId, long recoverySeq) throws IOException {
        LogFileKey start = SortedLogRecovery.minKey(LogEvents.MUTATION, tabletId);
        start.seq = recoverySeq;
        LogFileKey end = SortedLogRecovery.maxKey(LogEvents.MUTATION, tabletId);
        try (RecoveryLogsIterator rli = new RecoveryLogsIterator(this.fs, recoveryLogs, start, end);){
            while (rli.hasNext()) {
                Object entry = rli.next();
                Preconditions.checkState((((LogFileKey)entry.getKey()).tabletId == tabletId ? 1 : 0) != 0);
                Preconditions.checkState((((LogFileKey)entry.getKey()).seq >= recoverySeq ? 1 : 0) != 0);
                if (((LogFileKey)entry.getKey()).event == LogEvents.MUTATION) {
                    mr.receive(((LogFileValue)entry.getValue()).mutations.get(0));
                    continue;
                }
                if (((LogFileKey)entry.getKey()).event == LogEvents.MANY_MUTATIONS) {
                    for (Mutation m : ((LogFileValue)entry.getValue()).mutations) {
                        mr.receive(m);
                    }
                    continue;
                }
                throw new IllegalStateException("Non mutation event seen " + (Object)((Object)((LogFileKey)entry.getKey()).event));
            }
        }
    }

    Collection<String> asNames(List<Path> recoveryLogs) {
        return Collections2.transform(recoveryLogs, Path::getName);
    }

    public void recover(KeyExtent extent, List<Path> recoveryLogs, Set<String> tabletFiles, MutationReceiver mr) throws IOException {
        Map.Entry<Integer, List<Path>> maxEntry = this.findLogsThatDefineTablet(extent, recoveryLogs);
        int tabletId = maxEntry.getKey();
        List<Path> logsThatDefineTablet = maxEntry.getValue();
        if (tabletId == -1) {
            log.info("Tablet {} is not defined in recovery logs {} ", (Object)extent, this.asNames(recoveryLogs));
            return;
        }
        log.info("Found {} of {} logs with max id {} for tablet {}", new Object[]{logsThatDefineTablet.size(), recoveryLogs.size(), tabletId, extent});
        long recoverySeq = this.findRecoverySeq(logsThatDefineTablet, tabletFiles, tabletId);
        log.info("Recovering mutations, tablet:{} tabletId:{} seq:{} logs:{}", new Object[]{extent, tabletId, recoverySeq, this.asNames(logsThatDefineTablet)});
        this.playbackMutations(logsThatDefineTablet, mr, tabletId, recoverySeq);
    }

    static class DeduplicatingIterator
    implements Iterator<Map.Entry<LogFileKey, LogFileValue>> {
        private PeekingIterator<Map.Entry<LogFileKey, LogFileValue>> source;

        public DeduplicatingIterator(Iterator<Map.Entry<LogFileKey, LogFileValue>> source) {
            this.source = Iterators.peekingIterator(source);
        }

        @Override
        public boolean hasNext() {
            return this.source.hasNext();
        }

        @Override
        public Map.Entry<LogFileKey, LogFileValue> next() {
            Map.Entry next = (Map.Entry)this.source.next();
            while (this.source.hasNext() && ((LogFileKey)next.getKey()).compareTo((LogFileKey)((Map.Entry)this.source.peek()).getKey()) == 0) {
                this.source.next();
            }
            return next;
        }
    }
}

