/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class FileLogInputStreamTest {
    private final byte magic;
    private final CompressionType compression;

    public FileLogInputStreamTest(byte magic, CompressionType compression) {
        this.magic = magic;
        this.compression = compression;
    }

    @Test
    public void testWriteTo() throws IOException {
        if (this.compression == CompressionType.ZSTD && this.magic < 2) {
            return;
        }
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (CompressionType)this.compression, (SimpleRecord[])new SimpleRecord[]{new SimpleRecord("foo".getBytes())}));
            fileRecords.flush();
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, fileRecords.sizeInBytes());
            FileLogInputStream.FileChannelRecordBatch batch = logInputStream.nextBatch();
            Assert.assertNotNull((Object)batch);
            Assert.assertEquals((long)this.magic, (long)batch.magic());
            ByteBuffer buffer = ByteBuffer.allocate(128);
            batch.writeTo(buffer);
            buffer.flip();
            MemoryRecords memRecords = MemoryRecords.readableRecords((ByteBuffer)buffer);
            List records = Utils.toList(memRecords.records().iterator());
            Assert.assertEquals((long)1L, (long)records.size());
            Record record0 = (Record)records.get(0);
            Assert.assertTrue((boolean)record0.hasMagic(this.magic));
            Assert.assertEquals((Object)"foo", (Object)Utils.utf8((ByteBuffer)record0.value(), (int)record0.valueSize()));
        }
    }

    @Test
    public void testSimpleBatchIteration() throws IOException {
        if (this.compression == CompressionType.ZSTD && this.magic < 2) {
            return;
        }
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            SimpleRecord firstBatchRecord = new SimpleRecord(3241324L, "a".getBytes(), "foo".getBytes());
            SimpleRecord secondBatchRecord = new SimpleRecord(234280L, "b".getBytes(), "bar".getBytes());
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)0L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])new SimpleRecord[]{firstBatchRecord}));
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)1L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])new SimpleRecord[]{secondBatchRecord}));
            fileRecords.flush();
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, fileRecords.sizeInBytes());
            FileLogInputStream.FileChannelRecordBatch firstBatch = logInputStream.nextBatch();
            this.assertGenericRecordBatchData((RecordBatch)firstBatch, 0L, 3241324L, firstBatchRecord);
            this.assertNoProducerData((RecordBatch)firstBatch);
            FileLogInputStream.FileChannelRecordBatch secondBatch = logInputStream.nextBatch();
            this.assertGenericRecordBatchData((RecordBatch)secondBatch, 1L, 234280L, secondBatchRecord);
            this.assertNoProducerData((RecordBatch)secondBatch);
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    @Test
    public void testBatchIterationWithMultipleRecordsPerBatch() throws IOException {
        if (this.magic < 2 && this.compression == CompressionType.NONE) {
            return;
        }
        if (this.compression == CompressionType.ZSTD && this.magic < 2) {
            return;
        }
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            SimpleRecord[] firstBatchRecords = new SimpleRecord[]{new SimpleRecord(3241324L, "a".getBytes(), "1".getBytes()), new SimpleRecord(234280L, "b".getBytes(), "2".getBytes())};
            SimpleRecord[] secondBatchRecords = new SimpleRecord[]{new SimpleRecord(238423489L, "c".getBytes(), "3".getBytes()), new SimpleRecord(897839L, null, "4".getBytes()), new SimpleRecord(8234020L, "e".getBytes(), null)};
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)0L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])firstBatchRecords));
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)1L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])secondBatchRecords));
            fileRecords.flush();
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, fileRecords.sizeInBytes());
            FileLogInputStream.FileChannelRecordBatch firstBatch = logInputStream.nextBatch();
            this.assertNoProducerData((RecordBatch)firstBatch);
            this.assertGenericRecordBatchData((RecordBatch)firstBatch, 0L, 3241324L, firstBatchRecords);
            FileLogInputStream.FileChannelRecordBatch secondBatch = logInputStream.nextBatch();
            this.assertNoProducerData((RecordBatch)secondBatch);
            this.assertGenericRecordBatchData((RecordBatch)secondBatch, 1L, 238423489L, secondBatchRecords);
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    @Test
    public void testBatchIterationV2() throws IOException {
        if (this.magic != 2) {
            return;
        }
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            long producerId = 83843L;
            short producerEpoch = 15;
            int baseSequence = 234;
            int partitionLeaderEpoch = 9832;
            SimpleRecord[] firstBatchRecords = new SimpleRecord[]{new SimpleRecord(3241324L, "a".getBytes(), "1".getBytes()), new SimpleRecord(234280L, "b".getBytes(), "2".getBytes())};
            SimpleRecord[] secondBatchRecords = new SimpleRecord[]{new SimpleRecord(238423489L, "c".getBytes(), "3".getBytes()), new SimpleRecord(897839L, null, "4".getBytes()), new SimpleRecord(8234020L, "e".getBytes(), null)};
            fileRecords.append(MemoryRecords.withIdempotentRecords((byte)this.magic, (long)15L, (CompressionType)this.compression, (long)producerId, (short)producerEpoch, (int)baseSequence, (int)partitionLeaderEpoch, (SimpleRecord[])firstBatchRecords));
            fileRecords.append(MemoryRecords.withTransactionalRecords((byte)this.magic, (long)27L, (CompressionType)this.compression, (long)producerId, (short)producerEpoch, (int)(baseSequence + firstBatchRecords.length), (int)partitionLeaderEpoch, (SimpleRecord[])secondBatchRecords));
            fileRecords.flush();
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, fileRecords.sizeInBytes());
            FileLogInputStream.FileChannelRecordBatch firstBatch = logInputStream.nextBatch();
            this.assertProducerData((RecordBatch)firstBatch, producerId, producerEpoch, baseSequence, false, firstBatchRecords);
            this.assertGenericRecordBatchData((RecordBatch)firstBatch, 15L, 3241324L, firstBatchRecords);
            Assert.assertEquals((long)partitionLeaderEpoch, (long)firstBatch.partitionLeaderEpoch());
            FileLogInputStream.FileChannelRecordBatch secondBatch = logInputStream.nextBatch();
            this.assertProducerData((RecordBatch)secondBatch, producerId, producerEpoch, baseSequence + firstBatchRecords.length, true, secondBatchRecords);
            this.assertGenericRecordBatchData((RecordBatch)secondBatch, 27L, 238423489L, secondBatchRecords);
            Assert.assertEquals((long)partitionLeaderEpoch, (long)secondBatch.partitionLeaderEpoch());
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    @Test
    public void testBatchIterationIncompleteBatch() throws IOException {
        if (this.compression == CompressionType.ZSTD && this.magic < 2) {
            return;
        }
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            SimpleRecord firstBatchRecord = new SimpleRecord(100L, "foo".getBytes());
            SimpleRecord secondBatchRecord = new SimpleRecord(200L, "bar".getBytes());
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)0L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])new SimpleRecord[]{firstBatchRecord}));
            fileRecords.append(MemoryRecords.withRecords((byte)this.magic, (long)1L, (CompressionType)this.compression, (TimestampType)TimestampType.CREATE_TIME, (SimpleRecord[])new SimpleRecord[]{secondBatchRecord}));
            fileRecords.flush();
            fileRecords.truncateTo(fileRecords.sizeInBytes() - 13);
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, fileRecords.sizeInBytes());
            FileLogInputStream.FileChannelRecordBatch firstBatch = logInputStream.nextBatch();
            this.assertNoProducerData((RecordBatch)firstBatch);
            this.assertGenericRecordBatchData((RecordBatch)firstBatch, 0L, 100L, firstBatchRecord);
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    @Test
    public void testNextBatchSelectionWithMaxedParams() throws IOException {
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, Integer.MAX_VALUE, Integer.MAX_VALUE);
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    @Test
    public void testNextBatchSelectionWithZeroedParams() throws IOException {
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            FileLogInputStream logInputStream = new FileLogInputStream(fileRecords, 0, 0);
            Assert.assertNull((Object)logInputStream.nextBatch());
        }
    }

    private void assertProducerData(RecordBatch batch, long producerId, short producerEpoch, int baseSequence, boolean isTransactional, SimpleRecord ... records) {
        Assert.assertEquals((long)producerId, (long)batch.producerId());
        Assert.assertEquals((long)producerEpoch, (long)batch.producerEpoch());
        Assert.assertEquals((long)baseSequence, (long)batch.baseSequence());
        Assert.assertEquals((long)(baseSequence + records.length - 1), (long)batch.lastSequence());
        Assert.assertEquals((Object)isTransactional, (Object)batch.isTransactional());
    }

    private void assertNoProducerData(RecordBatch batch) {
        Assert.assertEquals((long)-1L, (long)batch.producerId());
        Assert.assertEquals((long)-1L, (long)batch.producerEpoch());
        Assert.assertEquals((long)-1L, (long)batch.baseSequence());
        Assert.assertEquals((long)-1L, (long)batch.lastSequence());
        Assert.assertFalse((boolean)batch.isTransactional());
    }

    private void assertGenericRecordBatchData(RecordBatch batch, long baseOffset, long maxTimestamp, SimpleRecord ... records) {
        Assert.assertEquals((long)this.magic, (long)batch.magic());
        Assert.assertEquals((Object)this.compression, (Object)batch.compressionType());
        if (this.magic == 0) {
            Assert.assertEquals((Object)TimestampType.NO_TIMESTAMP_TYPE, (Object)batch.timestampType());
        } else {
            Assert.assertEquals((Object)TimestampType.CREATE_TIME, (Object)batch.timestampType());
            Assert.assertEquals((long)maxTimestamp, (long)batch.maxTimestamp());
        }
        Assert.assertEquals((long)(baseOffset + (long)records.length - 1L), (long)batch.lastOffset());
        if (this.magic >= 2) {
            Assert.assertEquals((Object)records.length, (Object)batch.countOrNull());
        }
        Assert.assertEquals((long)baseOffset, (long)batch.baseOffset());
        Assert.assertTrue((boolean)batch.isValid());
        List batchRecords = TestUtils.toList(batch);
        for (int i = 0; i < records.length; ++i) {
            Assert.assertEquals((long)(baseOffset + (long)i), (long)((Record)batchRecords.get(i)).offset());
            Assert.assertEquals((Object)records[i].key(), (Object)((Record)batchRecords.get(i)).key());
            Assert.assertEquals((Object)records[i].value(), (Object)((Record)batchRecords.get(i)).value());
            if (this.magic == 0) {
                Assert.assertEquals((long)-1L, (long)((Record)batchRecords.get(i)).timestamp());
                continue;
            }
            Assert.assertEquals((long)records[i].timestamp(), (long)((Record)batchRecords.get(i)).timestamp());
        }
    }

    @Parameterized.Parameters(name="magic={0}, compression={1}")
    public static Collection<Object[]> data() {
        ArrayList<Object[]> values = new ArrayList<Object[]>();
        for (byte magic : Arrays.asList((byte)0, (byte)1, (byte)2)) {
            for (CompressionType type : CompressionType.values()) {
                values.add(new Object[]{magic, type});
            }
        }
        return values;
    }
}

