package org.openxri.admin.resources;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Stroke;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.wicket.Application;
import org.apache.wicket.markup.html.image.resource.RenderedDynamicImageResource;
import org.apache.wicket.markup.html.link.ImageMap;
import org.openxri.admin.webapplication.OpenXRIAdminApplication;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreBetterLookup;
import org.openxri.store.SubSegment;

public class AuthorityImageResource extends RenderedDynamicImageResource {

	private static final long serialVersionUID = -6682154691595772165L;

	protected static final int IMAGE_WIDTH = 800;
	protected static final int IMAGE_HEIGHT = 300;
	protected static final int IMAGE_PADDING_X = 35;
	protected static final int IMAGE_PADDING_Y = 35;
	protected static final int LEFT = 0 + IMAGE_PADDING_X;
	protected static final int TOP = 0 + IMAGE_PADDING_Y;
	protected static final int RIGHT = IMAGE_WIDTH - IMAGE_PADDING_X;
	protected static final int BOTTOM = IMAGE_HEIGHT - IMAGE_PADDING_Y;
	protected static final int WIDTH = RIGHT - LEFT;
	protected static final int HEIGHT = BOTTOM - TOP;
	protected static final int CENTER_X = IMAGE_WIDTH / 2;
	protected static final int CENTER_Y = IMAGE_HEIGHT / 2;
	protected static final int CENTER_AUTHORITY_WIDTH = 20;
	protected static final int CENTER_AUTHORITY_HEIGHT = 20;
	protected static final int AUTHORITY_WIDTH = 20;
	protected static final int AUTHORITY_HEIGHT = 20;

	protected static final Color CENTER_AUTHORITY_COLOR = new Color(3, 109, 167);
	protected static final Color AUTHORITY_COLOR = Color.DARK_GRAY;
	protected static final Color LINE_COLOR = Color.DARK_GRAY;
	protected static final Stroke LINE_STROKE = new BasicStroke(3);
	protected static final Color TEXT_COLOR = Color.WHITE;
	protected static final Font TEXT_FONT = new Font("Sans-serif", Font.PLAIN, 10);

	protected String authorityText;
	protected Map<String, List<String>> parentAuthorities;
	protected Map<String, List<String>> childAuthorities;

	public AuthorityImageResource(Authority authority) throws StoreException {

		super(IMAGE_WIDTH, IMAGE_HEIGHT, "png");

		final Store openXRIStore = ((OpenXRIAdminApplication) Application.get()).getOpenXRIStore();

		if (authority.getXrd().getCanonicalID() != null)
			this.authorityText = authority.getXrd().getCanonicalID().getValue();
		else
			this.authorityText = authority.getId().toString();

		SubSegment[] subSegments = ((StoreBetterLookup) openXRIStore).getAuthoritySubSegments(authority);
		SubSegment[] childSubSegments = ((StoreBetterLookup) openXRIStore).getAuthorityChildSubSegments(authority);

		this.parentAuthorities = new HashMap<String, List<String>> ();
		for (SubSegment subSegment : subSegments) {

			Authority parentAuthority = openXRIStore.getSubSegmentParentAuthority(subSegment);

			String authorityText;
			if (parentAuthority == null || parentAuthority.getXrd() == null)
				authorityText = "-";
			else if (parentAuthority.getXrd().getCanonicalID() != null)
				authorityText = parentAuthority.getXrd().getCanonicalID().getValue();
			else
				authorityText = parentAuthority.getId().toString();

			List<String> subSegmentNames = this.parentAuthorities.get(authorityText);
			if (subSegmentNames == null) {

				subSegmentNames = new ArrayList<String> ();
				this.parentAuthorities.put(authorityText, subSegmentNames);
			}
			subSegmentNames.add(subSegment.getName());
		}

		this.childAuthorities = new HashMap<String, List<String>> ();
		for (SubSegment childSubSegment : childSubSegments) {

			Authority childAuthority = openXRIStore.getSubSegmentAuthority(childSubSegment);

			String authorityText;
			if (childAuthority == null || childAuthority.getXrd() == null)
				authorityText = "-";
			else if (childAuthority.getXrd().getCanonicalID() != null)
				authorityText = childAuthority.getXrd().getCanonicalID().getValue();
			else
				authorityText = childAuthority.getId().toString();

			List<String> subSegmentNames = this.childAuthorities.get(authorityText);
			if (subSegmentNames == null) {

				subSegmentNames = new ArrayList<String> ();
				this.childAuthorities.put(authorityText, subSegmentNames);
			}
			subSegmentNames.add(childSubSegment.getName());
		}
	}

	@Override
	protected boolean render(Graphics2D graphics) {

		graphics.setFont(TEXT_FONT);

		renderParents(graphics);
		renderChildren(graphics);
		renderCenterAuthority(graphics);

		return(true);
	}

	protected void renderParents(Graphics2D graphics) {

		if (this.parentAuthorities.size() < 1) return;

		int subSegmentWidth = WIDTH / this.parentAuthorities.size();
		int subSegmentY = TOP;

		int i=0;
		for (Map.Entry<String, List<String>> entry : this.parentAuthorities.entrySet()) {

			renderAuthority(graphics, LEFT + (subSegmentWidth/2) + subSegmentWidth*i, subSegmentY, entry.getKey(), entry.getValue());
			i++;
		}
	}

	protected void renderChildren(Graphics2D graphics) {

		if (this.childAuthorities.size() < 1) return;

		int childSubSegmentWidth = WIDTH / this.childAuthorities.size();
		int childSubSegmentY = BOTTOM;

		int i=0;
		for (Map.Entry<String, List<String>> entry : this.childAuthorities.entrySet()) {

			renderAuthority(graphics, LEFT + (childSubSegmentWidth/2) + childSubSegmentWidth*i, childSubSegmentY, entry.getKey(), entry.getValue());
			i++;
		}
	}

	protected void renderCenterAuthority(Graphics2D graphics) {

		graphics.setColor(CENTER_AUTHORITY_COLOR);
		graphics.setBackground(CENTER_AUTHORITY_COLOR);
		graphics.fillOval(CENTER_X - CENTER_AUTHORITY_WIDTH / 2, CENTER_Y - CENTER_AUTHORITY_HEIGHT / 2, CENTER_AUTHORITY_WIDTH, CENTER_AUTHORITY_HEIGHT);

		write(graphics, this.authorityText, CENTER_X + 15, CENTER_Y - 5, false);
	}

	protected void renderAuthority(Graphics2D graphics, int x, int y, String authorityText, List<String> subSegmentNames) {

		int posX, posY;

		graphics.setColor(LINE_COLOR);
		graphics.setStroke(LINE_STROKE);
		graphics.drawLine(CENTER_X, CENTER_Y, x, y);

		posX = x - AUTHORITY_WIDTH / 2;
		posY = y - AUTHORITY_HEIGHT / 2;

		graphics.setColor(AUTHORITY_COLOR);
		graphics.setBackground(AUTHORITY_COLOR);
		if (authorityText.equals("-")) {

			graphics.drawOval(posX, posY, AUTHORITY_WIDTH, AUTHORITY_HEIGHT);
		} else {
		
			graphics.fillOval(posX, posY, AUTHORITY_WIDTH, AUTHORITY_HEIGHT);
			write(graphics, authorityText, x, y, true);
		}

		int subSegmentsOffset = (subSegmentNames.size() - 1) * 15;

		for (int i=0; i<subSegmentNames.size(); i++) {

			write(graphics, subSegmentNames.get(i), (x + CENTER_X) / 2, (y + CENTER_Y - subSegmentsOffset) / 2 + (i * 15), true);
		}
	}

	private static void write(Graphics2D graphics, String text, int posX, int posY, boolean center) {

		FontMetrics fontMetrics = graphics.getFontMetrics(TEXT_FONT);

		if (center) posX -= (int) (fontMetrics.getStringBounds(text, graphics).getWidth() / 2);
		if (center) posY -= (int) (fontMetrics.getStringBounds(text, graphics).getHeight() / 2);

		graphics.setColor(TEXT_COLOR);
		graphics.drawString(text, posX, posY + fontMetrics.getAscent());
	}

	public ImageMap createImageMap(String id) {

		ImageMap map = new ImageMap(id);

		return(map);
	}
}
