package org.openxri.admin.webpages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.wicket.Application;
import org.apache.wicket.behavior.HeaderContributor;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.TextField;
import org.apache.wicket.markup.html.link.BookmarkablePageLink;
import org.apache.wicket.markup.html.pages.RedirectPage;
import org.apache.wicket.markup.html.panel.FeedbackPanel;
import org.apache.wicket.model.CompoundPropertyModel;
import org.openxri.XRI;
import org.openxri.XRIAuthority;
import org.openxri.admin.webapplication.OpenXRIAdminApplication;
import org.openxri.admin.webpages.store.AuthorityDetails;
import org.openxri.exceptions.StoreException;
import org.openxri.exceptions.StoreNotFoundException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreBetterLookup;

public abstract class BasePage extends WebPage {

	private static final Log log = LogFactory.getLog(BasePage.class);

	private Label titleLabel;

	public BasePage() {

		// add css

		this.add(HeaderContributor.forCss("style.css", "screen"));

		// create and add components

		this.titleLabel = new Label("titleLabel", this.getClass().getName());

		this.add(new BookmarkablePageLink("homePageLink", Application.get().getHomePage()));
		this.add(this.titleLabel);
		this.add(new BasePageMenuPanel("menuPanel"));
		this.add(new FeedbackPanel("feedbackPanel"));
		this.add(new FindAuthorityForm("findAuthorityForm"));
		this.add(new QuickRetrievalForm("quickRetrievalForm"));
	}

	protected void setTitle(String title) {

		this.titleLabel.setModelObject(title);
	}

	private class FindAuthorityForm extends Form {

		private static final long serialVersionUID = 5652739836195398866L;

		private String xri;

		public FindAuthorityForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextField("xri"));
		}

		public String getXri() {
			return (this.xri);
		}

		public void setXri(String xri) {
			this.xri = xri;
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// find the authority

			Authority authority;

			try {

				XRI xri = new XRI(this.xri);
				XRIAuthority xriAuthority = ((XRIAuthority) xri.getAuthorityPath());

				authority = ((StoreBetterLookup) openXRIStore).localLookup(xriAuthority);

				if (authority == null) throw new StoreNotFoundException();
			} catch (StoreNotFoundException ex) {

				BasePage.log.error(ex);
				BasePage.this.error(BasePage.this.getString("notfound") + ex.getLocalizedMessage());
				return;
			} catch (StoreException ex) {

				BasePage.log.error(ex);
				BasePage.this.error(BasePage.this.getString("storefail") + ex.getLocalizedMessage());
				return;
			} catch (Exception ex) {

				BasePage.log.error(ex);
				BasePage.this.error(BasePage.this.getString("parsefail") + ex.getLocalizedMessage());
				return;
			}

			// go to authority details

			BasePage.this.setResponsePage(new AuthorityDetails(authority));
		}
	}

	private class QuickRetrievalForm extends Form {

		private static final long serialVersionUID = -7274158620417007644L;

		private String xri;

		public QuickRetrievalForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextField("xri"));
		}

		public String getXri() {
			return (this.xri);
		}
		public void setXri(String xri) {
			this.xri = xri;
		}

		@Override
		protected void onSubmit() {

			try {

				new XRI(this.xri);
			} catch (Exception ex) {

				BasePage.log.error(ex);
				BasePage.this.error(BasePage.this.getString("parsefail") + ex.getLocalizedMessage());
				return;
			}

			String hxri = ((OpenXRIAdminApplication) this.getApplication()).getProperties().getProperty("hxri-prefix") + this.xri;
			BasePage.this.setResponsePage(new RedirectPage(hxri + "?_xrd_r=application/xrds+xml;sep=false;debug=1"));
		}
	}
}
