package org.openxri.admin.webpages.store;

import java.io.StringReader;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.wicket.Page;
import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.ajax.markup.html.form.AjaxSubmitLink;
import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.DropDownChoice;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.TextArea;
import org.apache.wicket.markup.html.form.TextField;
import org.apache.wicket.markup.html.image.Image;
import org.apache.wicket.markup.html.link.IPageLink;
import org.apache.wicket.markup.html.link.PageLink;
import org.apache.wicket.markup.html.list.ListItem;
import org.apache.wicket.markup.html.list.ListView;
import org.apache.wicket.model.CompoundPropertyModel;
import org.apache.wicket.model.Model;
import org.openxri.admin.components.EmptyComponent;
import org.openxri.admin.components.MapPanel;
import org.openxri.admin.models.SEPTemplatesModel;
import org.openxri.admin.resources.AuthorityImageResource;
import org.openxri.admin.webapplication.OpenXRIAdminApplication;
import org.openxri.admin.webpages.BasePage;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.store.StoreBetterLookup;
import org.openxri.store.StoreEditable;
import org.openxri.store.StoreMountable;
import org.openxri.store.SubSegment;
import org.openxri.xml.Service;
import org.openxri.xml.XRD;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

public class AuthorityDetails extends BasePage {

	private static final long serialVersionUID = 1395259414379177607L;

	private static final Log log = LogFactory.getLog(AuthorityDetails.class);

	private final String _storefail = this.getString("storefail");
	private final String _attributessuccess = this.getString("attributessuccess");
	private final String _indexsuccess = this.getString("indexsuccess");
	private final String _pathsuccess = this.getString("pathsuccess");
	private final String _xrdfail = this.getString("xrdfail");
	private final String _xrdsuccess = this.getString("xrdsuccess");

	private Authority authority;

	private WebMarkupContainer subSegmentsContainer;
	private WebMarkupContainer childSubSegmentsContainer;
	
	private Long authorityId;
	private List<SubSegment> subSegments;
	private List<SubSegment> childSubSegments;
	private List<String> qxris;

	public AuthorityDetails(Authority authority) {

		if (authority == null) throw new NullPointerException();
		
		this.setTitle(this.getString("title"));
		this.setModel(new CompoundPropertyModel(this));

		this.authority = authority;

		// create and add components

		try {

			AuthorityImageResource graph = new AuthorityImageResource(authority);
			this.add(new Image("graph", graph));
		} catch (StoreException ex) {

			this.add(new EmptyComponent("graph"));
		}

		this.add(new Label("authorityId"));
		this.add(new ListView("qxris") {

			private static final long serialVersionUID = -7522679699522835198L;

			@Override
			protected void populateItem(ListItem item) {

				String qxri = item.getModelObjectAsString();

				item.add(new Label("qxri", qxri));
			}
		});

		this.add(new PageLink("createSubSegmentLink", new IPageLink() {

			private static final long serialVersionUID = 2192286434779574430L;

			public Page getPage() {

				return(new CreateSubSegment(AuthorityDetails.this.authority));
			}

			public Class<?> getPageIdentity() {

				return(CreateSubSegment.class);
			}
		}));

		this.subSegmentsContainer = new WebMarkupContainer("subSegmentsContainer");
		this.childSubSegmentsContainer = new WebMarkupContainer("childSubSegmentsContainer");
		
		this.subSegmentsContainer.add(new ListView("subSegments") {

			private static final long serialVersionUID = 8793019471745319174L;

			@Override
			protected void populateItem(ListItem item) {

				final SubSegment subSegment = (SubSegment) item.getModelObject();

				PageLink link = new PageLink("link", new IPageLink() {

					private static final long serialVersionUID = 4182879013901656829L;

					public Page getPage() {

						return(new SubSegmentDetails(subSegment));
					}

					public Class<?> getPageIdentity() {

						return(SubSegmentDetails.class);
					}
				});
				link.add(new Label("name", subSegment.getName()));
				item.add(link);
			}
		});
		this.childSubSegmentsContainer.add(new ListView("childSubSegments") {

			private static final long serialVersionUID = 1273612869821409740L;

			@Override
			protected void populateItem(ListItem item) {

				final SubSegment subSegment = (SubSegment) item.getModelObject();

				PageLink link = new PageLink("link", new IPageLink() {

					private static final long serialVersionUID = -1232152334705755201L;

					public Page getPage() {

						return(new SubSegmentDetails(subSegment));
					}

					public Class<?> getPageIdentity() {

						return(SubSegmentDetails.class);
					}
				});
				link.add(new Label("name", subSegment.getName()));
				item.add(link);
			}
		});
		this.add(this.subSegmentsContainer);
		this.add(this.childSubSegmentsContainer);
		this.add(new XrdForm("xrdForm"));
		this.add(new TemplateForm("templateForm"));
		this.add(new AttributesForm("attributesForm"));
		this.add(new IndexForm("indexForm"));
		this.add(new MountForm("mountForm"));
		this.add(new UnmountForm("unmountForm"));
		this.add(new DeleteForm("deleteForm"));
		this.add(new PageLink("backLink", Index.class));
	}

	@Override
	public void onBeforeRender() {

		final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

		try {

			this.authorityId = this.authority.getId();
			this.subSegments = Arrays.asList(((StoreBetterLookup) openXRIStore).getAuthoritySubSegments(this.authority));
			this.childSubSegments = Arrays.asList(((StoreBetterLookup) openXRIStore).getAuthorityChildSubSegments(this.authority));
			this.qxris = Arrays.asList(((StoreBetterLookup) openXRIStore).getAuthorityQxris(this.authority));
			
			AuthorityDetails.this.subSegmentsContainer.setVisible(this.subSegments.size() > 0);
			AuthorityDetails.this.childSubSegmentsContainer.setVisible(this.childSubSegments.size() > 0);
		} catch (StoreException ex) {

			log.error(ex);
			this.error(this._storefail + ex.getLocalizedMessage());
		}

		super.onBeforeRender();
	}

	public Long getAuthorityId() {
		return (this.authorityId);
	}
	public void setAuthorityId(Long authorityId) {
		this.authorityId = authorityId;
	}
	public List<SubSegment> getChildSubSegments() {
		return (this.childSubSegments);
	}
	public void setChildSubSegments(List<SubSegment> childSubSegments) {
		this.childSubSegments = childSubSegments;
	}
	public List<String> getQxris() {
		return (this.qxris);
	}
	public void setQxris(List<String> qxris) {
		this.qxris = qxris;
	}
	public List<SubSegment> getSubSegments() {
		return (this.subSegments);
	}
	public void setSubSegments(List<SubSegment> subSegments) {
		this.subSegments = subSegments;
	}

	private class XrdForm extends Form {

		private static final long serialVersionUID = -2048988947965861549L;

		private String xrdString;

		public XrdForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextArea("xrdString"));
		}

		@Override
		protected void onBeforeRender() {

			// get information from the store

			this.xrdString = AuthorityDetails.this.authority.getXrd().toString();

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
				DocumentBuilder builder = factory.newDocumentBuilder();
				Document document = builder.parse(new InputSource(new StringReader(this.xrdString)));

				XRD xrd = new XRD();
				xrd.fromDOM(document.getDocumentElement());

				((StoreEditable) openXRIStore).setXrd(AuthorityDetails.this.authority, xrd);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			} catch (Exception ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._xrdfail + ex.getLocalizedMessage());
				return;
			}

			AuthorityDetails.this.info(AuthorityDetails.this._xrdsuccess);
		}

		public String getXrdString() {
			return (this.xrdString);
		}
		public void setXrdString(String xrdString) {
			this.xrdString = xrdString;
		}
	}

	private class TemplateForm extends Form {

		private static final long serialVersionUID = 5808914017160935468L;

		private String service;

		private DropDownChoice templateDropDownChoice;
		private AjaxSubmitLink loadTemplateButton;
		private TextArea serviceTextArea;

		public TemplateForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create components

			SEPTemplatesModel importSEPTemplatesModel = new SEPTemplatesModel();

			this.serviceTextArea = new TextArea("service");
			this.serviceTextArea.setOutputMarkupId(true);
			this.templateDropDownChoice = new DropDownChoice("template", new Model(new Service()), importSEPTemplatesModel, importSEPTemplatesModel); 
			this.loadTemplateButton = new AjaxSubmitLink("loadTemplateButton", this) {

				private static final long serialVersionUID = -4311234326704642467L;

				@Override
				protected void onSubmit(AjaxRequestTarget target, Form form) {

					if (TemplateForm.this.templateDropDownChoice.getModelObject() == null) return;

					TemplateForm.this.service = TemplateForm.this.templateDropDownChoice.getModelObject().toString();

					target.addComponent(TemplateForm.this.serviceTextArea);
				}
			};

			// add components

			this.add(this.templateDropDownChoice);
			this.add(this.loadTemplateButton);
			this.add(this.serviceTextArea);
		}

		public String getService() {
			return (this.service);
		}
		public void setService(String service) {
			this.service = service;
		}
	}

	private class AttributesForm extends Form {

		private static final long serialVersionUID = -4039852015387356990L;

		private Map<String, String> attributes;

		public AttributesForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new MapPanel("attributes"));
		}

		@Override
		protected void onBeforeRender() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// get information from the store

			try {

				this.attributes = ((StoreAttributable) openXRIStore).getAuthorityAttributes(AuthorityDetails.this.authority);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
			}

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreAttributable) openXRIStore).setAuthorityAttributes(AuthorityDetails.this.authority, this.attributes);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			AuthorityDetails.this.info(AuthorityDetails.this._attributessuccess);
		}

		public Map<String, String> getAttributes() {
			return (this.attributes);
		}
		public void setAttributes(Map<String, String> attributes) {
			this.attributes = attributes;
		}
	}

	private class IndexForm extends Form {

		private static final long serialVersionUID = -2048988947965861549L;

		private String index;

		public IndexForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextField("index"));
		}

		@Override
		protected void onBeforeRender() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// get information from the store

			try {

				this.index = ((StoreAttributable) openXRIStore).getAuthorityIndex(AuthorityDetails.this.authority);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
			}

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreAttributable) openXRIStore).setAuthorityIndex(AuthorityDetails.this.authority, this.index);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			AuthorityDetails.this.info(AuthorityDetails.this._indexsuccess);
		}

		public String getIndex() {
			return (this.index);
		}
		public void setIndex(String index) {
			this.index = index;
		}
	}

	private class MountForm extends Form {

		private static final long serialVersionUID = -3978666310705447013L;

		private String pth;

		public MountForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextField("pth"));
		}

		@Override
		protected void onBeforeRender() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// get information from the store

			try {

				this.pth = ((StoreMountable) openXRIStore).getAuthorityPath(AuthorityDetails.this.authority);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
			}

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreMountable) openXRIStore).setAuthorityPath(AuthorityDetails.this.authority, this.pth);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			AuthorityDetails.this.info(AuthorityDetails.this._pathsuccess);
		}

		public String getPth() {
			return (this.pth);
		}
		public void setPth(String pth) {
			this.pth = pth;
		}
	}

	private class UnmountForm extends Form {

		private static final long serialVersionUID = -8318342666005895301L;

		public UnmountForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreMountable) openXRIStore).setAuthorityPath(AuthorityDetails.this.authority, null);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			AuthorityDetails.this.info(AuthorityDetails.this._pathsuccess);
		}
	}

	private class DeleteForm extends Form {

		private static final long serialVersionUID = -1800953543119169631L;

		public DeleteForm(String id) {

			super(id);
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// delete the authority

			try {

				openXRIStore.deleteAuthority(AuthorityDetails.this.authority);
			} catch (StoreException ex) {

				AuthorityDetails.log.error(ex);
				AuthorityDetails.this.error(AuthorityDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			// back to authority list

			this.setResponsePage(ListAuthorities.class);
		}
	}
}
