package org.openxri.admin.webpages.store;

import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.wicket.Page;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.TextField;
import org.apache.wicket.markup.html.image.Image;
import org.apache.wicket.markup.html.link.IPageLink;
import org.apache.wicket.markup.html.link.PageLink;
import org.apache.wicket.model.CompoundPropertyModel;
import org.openxri.admin.components.EmptyComponent;
import org.openxri.admin.components.MapPanel;
import org.openxri.admin.resources.SubSegmentImageResource;
import org.openxri.admin.webapplication.OpenXRIAdminApplication;
import org.openxri.admin.webpages.BasePage;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.store.SubSegment;

public class SubSegmentDetails extends BasePage {

	private static final Log log = LogFactory.getLog(SubSegmentDetails.class);

	private static final long serialVersionUID = -4904355435731649511L;

	private final String _storefail = this.getString("storefail");
	private final String _attributessuccess = this.getString("attributessuccess");
	private final String _indexsuccess = this.getString("indexsuccess");

	private SubSegment subSegment;

	private Long subSegmentId;
	private String name;
	private Authority parentAuthority;
	private Authority authority;

	public SubSegmentDetails(SubSegment subSegment) {

		if (subSegment == null) throw new NullPointerException();
		
		this.setTitle(this.getString("title"));
		this.setModel(new CompoundPropertyModel(this));

		final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

		this.subSegment = subSegment;

		// get information from the store

		try {

			this.parentAuthority = openXRIStore.getSubSegmentParentAuthority(this.subSegment);
			this.authority = openXRIStore.getSubSegmentAuthority(this.subSegment);
		} catch (StoreException ex) {

			log.error(ex);
			this.error(this._storefail + ex.getLocalizedMessage());
		}

		// create and add components

		try {

			SubSegmentImageResource graph = new SubSegmentImageResource(subSegment);
			this.add(new Image("graph", graph));
		} catch (StoreException ex) {

			this.add(new EmptyComponent("graph"));
		}

		this.add(new Label("subSegmentId"));
		this.add(new Label("name"));
		this.add(new PageLink("createAliasSubSegmentLink", new IPageLink() {

			private static final long serialVersionUID = 4603762735183925968L;

			public Page getPage() {

				return(new CreateAliasSubSegment(SubSegmentDetails.this.parentAuthority, SubSegmentDetails.this.authority));
			}

			public Class<?> getPageIdentity() {

				return(CreateAliasSubSegment.class);
			}
		}));
		
		if (this.authority != null) {

			this.add(new PageLink("authorityLink", new IPageLink() {

				private static final long serialVersionUID = 1L;

				public Page getPage() {

					return(new AuthorityDetails(SubSegmentDetails.this.authority));
				}

				public Class<?> getPageIdentity() {

					return(SubSegmentDetails.class);
				}
			}));
		} else {

			this.add(new EmptyComponent("authorityLink"));
		}

		if (this.parentAuthority != null) {

			this.add(new PageLink("parentAuthorityLink", new IPageLink() {

				private static final long serialVersionUID = 1L;

				public Page getPage() {

					return(new AuthorityDetails(SubSegmentDetails.this.parentAuthority));
				}

				public Class<?> getPageIdentity() {

					return(SubSegmentDetails.class);
				}
			}));
		} else {

			this.add(new EmptyComponent("parentAuthorityLink"));
		}
		this.add(new AttributesForm("attributesForm"));
		this.add(new IndexForm("indexForm"));
		this.add(new DeleteForm("deleteForm"));
		this.add(new PageLink("backLink", Index.class));
	}

	@Override
	protected void onBeforeRender() {

		final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

		// get information from the store

		try {

			this.subSegmentId = this.subSegment.getId();
			this.name = this.subSegment.getName();
			this.parentAuthority = openXRIStore.getSubSegmentParentAuthority(this.subSegment);
			this.authority = openXRIStore.getSubSegmentAuthority(this.subSegment);
		} catch (StoreException ex) {

			log.error(ex);
			this.error(this._storefail + ex.getLocalizedMessage());
		}

		super.onBeforeRender();
	}

	public Long getSubSegmentId() {
		return (this.subSegmentId);
	}
	public void setSubSegmentId(Long id) {
		this.subSegmentId = id;
	}
	public Authority getAuthority() {
		return (this.authority);
	}
	public void setAuthority(Authority authority) {
		this.authority = authority;
	}
	public String getName() {
		return (this.name);
	}
	public void setName(String name) {
		this.name = name;
	}
	public Authority getParentAuthority() {
		return (this.parentAuthority);
	}
	public void setParentAuthority(Authority parentAuthority) {
		this.parentAuthority = parentAuthority;
	}

	private class AttributesForm extends Form {

		private static final long serialVersionUID = -4039852015387356990L;

		private Map<String, String> attributes;

		public AttributesForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new MapPanel("attributes"));
		}

		@Override
		protected void onBeforeRender() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// get information from the store

			try {

				this.attributes = ((StoreAttributable) openXRIStore).getSubSegmentAttributes(SubSegmentDetails.this.subSegment);
			} catch (StoreException ex) {

				SubSegmentDetails.log.error(ex);
				SubSegmentDetails.this.error(SubSegmentDetails.this._storefail + ex.getLocalizedMessage());
			}

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreAttributable) openXRIStore).setSubSegmentAttributes(SubSegmentDetails.this.subSegment, this.attributes);
			} catch (StoreException ex) {

				SubSegmentDetails.log.error(ex);
				SubSegmentDetails.this.error(SubSegmentDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			SubSegmentDetails.this.info(SubSegmentDetails.this._attributessuccess);
		}

		public Map<String, String> getAttributes() {
			return (this.attributes);
		}
		public void setAttributes(Map<String, String> attributes) {
			this.attributes = attributes;
		}
	}

	private class IndexForm extends Form {

		private static final long serialVersionUID = 1499748684603593990L;

		private String index;

		public IndexForm(String id) {

			super(id);
			this.setModel(new CompoundPropertyModel(this));

			// create and add components

			this.add(new TextField("index"));
		}

		@Override
		protected void onBeforeRender() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// get information from the store

			try {

				this.index = ((StoreAttributable) openXRIStore).getSubSegmentIndex(SubSegmentDetails.this.subSegment);
			} catch (StoreException ex) {

				SubSegmentDetails.log.error(ex);
				SubSegmentDetails.this.error(SubSegmentDetails.this._storefail + ex.getLocalizedMessage());
			}

			super.onBeforeRender();
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// update information in the store

			try {

				((StoreAttributable) openXRIStore).setSubSegmentIndex(SubSegmentDetails.this.subSegment, this.index);
			} catch (StoreException ex) {

				SubSegmentDetails.log.error(ex);
				SubSegmentDetails.this.error(SubSegmentDetails.this._storefail + ex.getLocalizedMessage());
				return;
			}

			SubSegmentDetails.this.info(SubSegmentDetails.this._indexsuccess);
		}

		public String getIndex() {
			return (this.index);
		}
		public void setIndex(String index) {
			this.index = index;
		}
	}

	private class DeleteForm extends Form {

		private static final long serialVersionUID = 8688706855193581490L;

		public DeleteForm(String id) {

			super(id);
		}

		@Override
		protected void onSubmit() {

			final Store openXRIStore = ((OpenXRIAdminApplication) this.getApplication()).getOpenXRIStore();

			// delete the subsegment

			try {

				openXRIStore.releaseSubSegment(SubSegmentDetails.this.subSegment);
			} catch (StoreException ex) {

				SubSegmentDetails.log.error(ex);
				SubSegmentDetails.this.error(SubSegmentDetails.this._storefail + ex.getLocalizedMessage());
			}

			// back to subsegment list

			this.setResponsePage(ListSubSegments.class);
		}
	}
}
