/*
 * Copyright 2005 OpenXRI Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/
package org.openxri.resolve;

import java.io.Serializable;
import net.sf.ehcache.*;


public class ResolverCache
{
	/** default cache name */
	static final protected String CACHE_NAME = "xrires";


	/** cache manager instance */
	static protected CacheManager cacheManager = null;


	/** cache name */
	protected String cacheName = CACHE_NAME;


	/**
	 * default constructor
	 */
	public ResolverCache()
	{
		cacheManager = CacheManager.getInstance();
	}

	
	/**
	 * constructor initialized with cache manager instance
	 * @param mgr cache manager instance
	 */
	public ResolverCache(CacheManager mgr)
	{
		cacheManager = mgr;
	}


	/**
	 * replace current internal cache with a newly-created cache instance
	 * @param name cacheName
	 * @param maxElements maximum number of cached elements in memory
	 */
	public void setNewCache(String name, int maxElements)
	{
		Cache memoryOnlyCache = new Cache(name, maxElements, false, false, 5, 0);
		cacheManager.addCache(memoryOnlyCache);
		cacheName = name;
	}


	/**
	 * attempt to get value of the given authority from cache
	 * @param xriAuthority authority for which to lookup
	 * @param isHttps isHttps flag lookup key (used in conjunction with xriAuthority)
	 * @param isSaml isSaml flag lookup key (used in conjunction with xriAuthority)
	 * @return null if not in cache or expired, otherwise the XRDS buffer
	 */
	public byte[] get(String xriAuthority, boolean isHttps, boolean isSaml)
	{
		Cache cache = cacheManager.getCache(cacheName);
		if (cache == null)
			return null;

		Element e = cache.get(computeKey(xriAuthority, isHttps, isSaml));
		if (e == null)
			return null;
		// System.out.println(e.toString());
		// System.out.println(e.getExpirationTime());

		return (byte[])e.getValue();
	}


	/**
	 * store the given authority into the cache along with the resolved XRDS document
	 * @param xriAuthority authority for which to store
	 * @param isHttps isHttps flag key (used in conjunction with xriAuthority)
	 * @param isSaml isSaml flag key (used in conjunction with xriAuthority)
	 * @param xrdsBuf XRDS buffer to store
	 * @param ttl time to live in seconds
	 */
	public void put(String xriAuthority, boolean isHttps, boolean isSaml, byte[] xrdsBuf, int ttl)
	{
		Cache cache = cacheManager.getCache(cacheName);
		Serializable key = computeKey(xriAuthority, isHttps, isSaml);
		Element e = new Element(key, xrdsBuf);
		e.setTimeToLive(ttl);
		cache.put(e);
	}


	/**
	 * remove the given authority from the cache
	 * @param xriAuthority authority for which to remove
	 * @param isHttps isHttps flag key (used in conjunction with xriAuthority)
	 * @param isSaml isSaml flag key (used in conjunction with xriAuthority)
	 */
	public void del(String xriAuthority, boolean isHttps, boolean isSaml)
	{
		Cache cache = cacheManager.getCache(cacheName);
		Serializable key = computeKey(xriAuthority, isHttps, isSaml);
		cache.remove(key);
	}


	/**
	 * empty the cache
	 */
	public void clear()
	{
		Cache cache = cacheManager.getCache(cacheName);
		cache.removeAll(false);
	}


	/**
	 * get the maximum number of cached elements
	 * @return 
	 */
	public int getMaxSize()
	{
		Cache cache = cacheManager.getCache(cacheName);
		return cache.getMaxElementsInMemory();
	}


	/**
	 * get the number of cached elements currently cached
	 * @return
	 */
	public int getSize()
	{
		Cache cache = cacheManager.getCache(cacheName);
		return cache.getSize();
	}


	/**
	 * compute the string key from the various components, used for storage and lookup
	 * @param xriAuthority authority component
	 * @param isHttps isHttps component
	 * @param isSaml isSaml component
	 * @return key representing the components above combined
	 */
	protected static Serializable computeKey(String xriAuthority, boolean isHttps, boolean isSaml)
	{
		StringBuffer sb = new StringBuffer();
		sb.append(xriAuthority);
		sb.append(isHttps? "1":"0");
		sb.append(isSaml? "1": "0");
		return sb.toString();
	}


}
