/*
 * Copyright 2005 OpenXRI Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openxri.resolve;


import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;

import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.message.BasicHttpResponse;
import org.openxri.AuthorityPath;
import org.openxri.GCSAuthority;
import org.openxri.resolve.ResolverCache;
import org.openxri.util.ResolvedHttpResponse;
import org.openxri.xml.Service;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;


import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;



/**
 * @author =wil
 */
public class ResolverCacheTest extends TestCase
{
	public static final int ALLOW_JITTER = 5; // 5 seconds error allowance
	// not thread-safe but ok for unit tests
	public static final DateFormat dfm = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");

	public static void main(String[] oArgs)
	{
		TestRunner.run(suite());
	}

	public static Test suite()
	{
		return new TestSuite(ResolverCacheTest.class);
	}

	public void testCache()
	{
		ResolverCache cache = new ResolverCache();
		cache.setNewCache("test", 1000);
		assertTrue("Initial cache not empty", cache.getSize() == 0);

		XRD xrd = new XRD();
		Service atAuthService = new Service();
		atAuthService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none");
		atAuthService.addType(Tags.SERVICE_AUTH_RES);
		atAuthService.addURI("http://gcs.epok.net/xri/resolve?ns=at");
		xrd.addService(atAuthService);

		XRD xrd1 = new XRD();
		Service dummyService = new Service();
		dummyService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none");
		dummyService.addType(Tags.SERVICE_AUTH_RES);
		dummyService.addURI("http://www.example.com/xri/resolve?id=1");
		xrd1.addService(dummyService);

		try {
			GCSAuthority auth = new GCSAuthority("@");
			cache.put(auth.toString(), false, false, xrd.toString().getBytes("UTF-8"), 1000);
			assertTrue("Initial cache incorrect", cache.getSize() == 1);

			cache.put(
					AuthorityPath.buildAuthorityPath("@!a!b!foo").toString(), false, false, xrd1.toString().getBytes("UTF-8"), 1000);
			assertTrue("Cache size incorrect", cache.getSize() == 2);

			cache.put(
					AuthorityPath.buildAuthorityPath("@!a!b!foo").toString(), false, false, xrd1.toString().getBytes("UTF-8"), 1000);
			assertTrue("Cache stores duplicate element?", cache.getSize() == 2);


			byte[] val =
				cache.get(AuthorityPath.buildAuthorityPath("@!a!b!foo").toString(), false, false);
			assertTrue("Cached value not found", val != null);

			val =
				cache.get(AuthorityPath.buildAuthorityPath("@!a!b!woo").toString(), false, false);
			assertTrue("Cached value should not have been found", val == null);
		}
		catch (UnsupportedEncodingException e) {
			assertTrue("Unexpected exception" + e, false);
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	/// basic tests - no http response headers

	public void testBasicXrdNull()
	{
		// basic - XRD expires null
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is);
		final Calendar cal = Calendar.getInstance();
		
		// basic - XRD expires null
		(new Resolver() {
			{
				minCacheTTL = 5;
				defaultCacheTTL = 300;
			}
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("defaultCacheTTL not applied ttl(" + ttl + ") != defaultCacheTTL (" + defaultCacheTTL + ")",
						ttl == defaultCacheTTL);
			}
		}).doTest(null, rresp);
	}


	
	public void testBasicXrdExpired()
	{
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is);
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.SECOND, -60);
		(new Resolver() {
			{
				minCacheTTL = 5;
			}
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[basic-exp] minCacheTTL not respected ttl(" + ttl + ") != minCacheTTL (" + minCacheTTL + ")",
						ttl == minCacheTTL);
			}
		}).doTest(cal.getTime(), rresp);
	}

	
	public void testBasicXrd()
	{
		// basic - XRD expires in 10 minutes
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is);
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 10);
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[basic-10min] computed TTL (" + ttl + ") should be roughly 10 minutes",
						Math.abs(10*60 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);
	}

	
	public void testExpiresHttpExpires()
	{
		// XRD Expires null, HTTP Expires present
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 5);
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[http] computed TTL (" + ttl + ") should be roughly 5 minutes",
						Math.abs(5*60 - ttl) < ALLOW_JITTER);
			}
		}).doTest(null, rresp);
	}
	

	public void testExpiresHttpExpired()
	{
		// XRD Expires null, HTTP Expires expired
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, -9); // 9 minutes ago
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		(new Resolver() {
			{
				minCacheTTL = 5;
			}
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[http-exp] minCacheTTL not respected ttl(" + ttl + ") != minCacheTTL (" + minCacheTTL + ")",
						ttl == minCacheTTL);
			}
		}).doTest(null, rresp);
	}


	public void testExpiresXrdHttp()
	{
		// XRD and HTTP headers agree
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 5);

		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[expired] computed TTL (" + ttl + ") should be roughly 5 minutes",
						Math.abs(5*60 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);
	}

	
	public void testExpiresXrdHttpDiffer1()
	{
		// XRD and HTTP headers disagree (xrd > http)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 5);
		
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		cal.add(Calendar.MINUTE, 10); // add 10 minutes to XRD Expires
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[xrd>http] computed TTL (" + ttl + ") should be roughly 5 minutes",
						Math.abs(5*60 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);
	}
	
	
	public void testExpiresXrdHttpDiffer2()
	{
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 15); // 15 minutes

		// XRD and HTTP headers disagree (http > xrd)
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		cal.setTimeInMillis(System.currentTimeMillis() + 1000*60*5); // 5 minutes
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[http>xrd] computed TTL (" + ttl + ") should be roughly 5 minutes",
						Math.abs(5*60 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);		
	}
	

	public void testCacheControlNoCache()
	{
		// Cache-Control: no-cache
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.HOUR, 1);
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "no-cache");
			}
		};
		(new Resolver() {
			{ minCacheTTL = 123; }
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-nocache] computed TTL (" + ttl + ") should be equal to minCacheTTL",
						ttl == minCacheTTL);
			}
		}).doTest(cal.getTime(), rresp);
	}

	public void testCacheControlPrivate()
	{
		// Cache-Control: private
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.HOUR, 1);
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "private");
			}
		};
		(new Resolver() {
			{ minCacheTTL = 123; }
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-private] computed TTL (" + ttl + ") should be equal to minCacheTTL",
						ttl == minCacheTTL);
			}
		}).doTest(cal.getTime(), rresp);
	}


	public void testCacheControlMaxAge()
	{
		// Cache-Control: max-age=3000 (no xrd-expires, no http-expires)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "max-age=\"3000\"");
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-maxage] computed TTL (" + ttl + ") should be roughly 3000 secs",
						Math.abs(3000 - ttl) < ALLOW_JITTER);
			}
		}).doTest(null, rresp);
	}

	
	public void testCacheControlSMaxAge()
	{
		// Cache-Control: s-maxage=3000 (no xrd-expires, no http-expires)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "s-maxage=3000");
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-s-maxage] computed TTL (" + ttl + ") should be roughly 3000 secs",
						Math.abs(3000 - ttl) < ALLOW_JITTER);
			}
		}).doTest(null, rresp);
	}
	

	public void testCacheControlMaxAgeBoth1()
	{
		// Cache-Control: differing max-age and s-maxage (no xrd-expires, no http-expires)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "max-age=3000");
				resp.addHeader("Cache-Control", "s-maxage=600");
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-s-maxage-over] computed TTL (" + ttl + ") should be roughly 600 secs",
						Math.abs(600 - ttl) < ALLOW_JITTER);
			}
		}).doTest(null, rresp);
	}

	
	public void testCacheControlMaxAgeBoth2()
	{
		// Cache-Control: differing max-age and s-maxage (no xrd-expires, no http-expires)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "max-age=3000");
				resp.addHeader("Cache-Control", "s-maxage=6000");
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-s-maxage-over2] computed TTL (" + ttl + ") should be roughly 6000 secs",
						Math.abs(6000 - ttl) < ALLOW_JITTER);
			}
		}).doTest(null, rresp);
	}


	public void testCacheControlMaxAgeXrd()
	{
		// Cache-Control: differing max-age and http expires (max-age should override)
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, 61);

		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "max-age=4004");
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-maxage-over] computed TTL (" + ttl + ") should be roughly 4004 secs",
						Math.abs(4004 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);
	}

	
	public void testCacheControlMaxAgePublic()
	{
		// Cache-Control: public, max-age=3000, xrd-expires expired, http-expires expired
		// max-age should override
		ByteArrayInputStream is = new ByteArrayInputStream("dummy".getBytes());
		final Calendar cal = Calendar.getInstance();
		cal.add(Calendar.MINUTE, -60);
		ResolvedHttpResponse rresp = new ResolvedHttpResponse(is) {
			{
				resp = new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_OK, "OK");
				resp.addHeader("Cache-Control", "public,max-age=1800");
				resp.addHeader("Expires", dfm.format(cal.getTime()));
			}
		};
		(new Resolver() {
			public void doTest(Date xrdExpires, ResolvedHttpResponse response) {
				int ttl = computeCacheTTL(xrdExpires, response);
				assertTrue("[CC-maxage-public] computed TTL (" + ttl + ") should be roughly 1800 secs",
						Math.abs(1800 - ttl) < ALLOW_JITTER);
			}
		}).doTest(cal.getTime(), rresp);
	}



	public void testExpire()
	{
		ResolverCache c = new ResolverCache();
		try {
			c.del("@abc", false, false);
			byte[] s = c.get("@abc", false, false);
			assertTrue("stale cache value", s == null);
			
			c.put("@abc", false, false, "x".getBytes("UTF-8"), 2);
			s = c.get("@abc", false, false);
			assertTrue("unable to retrieve stored value", s != null);

			Thread.sleep(1000);
			s = c.get("@abc", false, false);
			assertTrue("premature expiry", s != null);
	
			Thread.sleep(1500);
			s = c.get("@abc", false, false);
			assertTrue("item should have expired", s == null);
		}
		catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			assertTrue(e.toString(), false);
		}
		catch (InterruptedException e) {
			e.printStackTrace();
			assertTrue(e.toString(), false);
		}
	}

	
	public void testConcurrent()
	{
		ResolverCache cache = new ResolverCache();
		cache.setNewCache("testConcurrent", 5);
		cache.del(AuthorityPath.buildAuthorityPath("@").toString(), false, false);
		assertTrue("Initial cache not empty", cache.getSize() == 0);


		XRD xrd = new XRD();
		Service atAuthService = new Service();
		atAuthService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none");
		atAuthService.addType(Tags.SERVICE_AUTH_RES);
		atAuthService.addURI("http://gcs.epok.net/xri/resolve?ns=at");
		xrd.addService(atAuthService);

		try {
			GCSAuthority auth = new GCSAuthority("@");
			cache.put(auth.toString(), false, false, xrd.toString().getBytes("UTF-8"), 1000);
			assertTrue("Initial cache incorrect", cache.getSize() == 1);
		} catch (UnsupportedEncodingException e) {
			assertTrue("Unexpected exception" + e, false);
		}

		Random rnd = new Random();

		try
		{
			Thread[] threads = new StuffPruneThread[100];
			for (int i = 0; i < threads.length; i++)
			{
				threads[i] = new StuffPruneThread(rnd, cache);
			}

			for (int i = 0; i < threads.length; i++)
			{
				threads[i].start();
			}

			for (int i = 0; i < threads.length; i++)
			{
				threads[i].join();
			}
		}
		catch (Exception e)
		{
			assertTrue("Unexpected exception" + e, false);
		}

		assertTrue(
				"Max cache size not honored",
				cache.getSize() <= cache.getMaxSize());


	}


	class StuffPruneThread extends Thread
	{
		private Random rnd = null;
		private ResolverCache cache = null;

		public StuffPruneThread(Random rnd, ResolverCache cache)
		{
			this.rnd = rnd;
			this.cache = cache;

		}

		public void run()
		{
			XRD xrd = new XRD();
			Service dummyService = new Service();
			dummyService.addMediaType(Tags.CONTENT_TYPE_XRDS + ";trust=none");
			dummyService.addType(Tags.SERVICE_AUTH_RES);
			dummyService.addURI("http://www.example.com/xri/resolve?id=1");
			xrd.addService(dummyService);

			String[] oCases =
			{ "@!a1!b2!c3!d4", "@!x1!y2!z3", "@!a1!b2!c3", "@!a1!b2", "@!a1!b2!m3", "@!a1!o2!p3", "@!a1!o2!q3", "@!a1!b2!c3!d4!e5", "@!x1!y2" };

			try {
				for (int i = 0; i < 1000; i++)
				{
					int x = rnd.nextInt(oCases.length);
					boolean bStuff = rnd.nextBoolean();
					String auth = AuthorityPath.buildAuthorityPath(oCases[x]).toString();

					if (bStuff) {
						cache.put(auth, false, false, xrd.toString().getBytes("UTF-8"), 1000);
					}
					else {
						cache.del(auth, false, false);
					}

					cache.get(auth, false, false);
				}
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

	}

}
