/*
 * Copyright 2005 OpenXRI Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/
package org.openxri.resolve;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xml.security.keys.KeyInfo;
import org.apache.xml.security.keys.content.X509Data;
import java.security.KeyStore;
import org.w3c.dom.Document;

import java.io.InputStream;
import java.util.ArrayList;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

import org.openxri.XRI;
import org.openxri.xml.CanonicalID;
import org.openxri.xml.Service;
import org.openxri.xml.Status;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;

import java.security.cert.X509Certificate;


public class TrustedResolver
	extends TestCase
{

      private static org.apache.commons.logging.Log log
		= org.apache.commons.logging.LogFactory.getLog(TrustedResolver.class.getName());

      public static void main(String[] oArgs)
	{
		// Pass control to the non-graphical test runner
		TestRunner.run(suite());
	}


    public static Test suite() {
		return new TestSuite(TrustedResolver.class);
	}


	public void testTrustedResolveCA() {
		String qxri, septype;

		try {
			Resolver resolver = setupResolver();
			qxri = "(@blog*lockbox)*peter2";
                        septype="xri://$certificate*($x.509)";
			XRI xri = XRI.fromIRINormalForm(qxri);

			ResolverFlags flags = new ResolverFlags();
			flags.setUric(true);
			flags.setHttps(false);
			flags.setSaml(true);

                        resolver.resolveSEPToXRD(xri, septype, null, flags, new ResolverState());

                        TrustType tt = new TrustType();
                        tt.setParameterPair(false, true);

			ArrayList uris = resolver.resolveSEPToURIList(qxri, tt, septype, null, true);


		}
		catch (Exception e) {
			e.printStackTrace();
			fail("Not expecting an exception here: " + e);
		}
	}

	public void testTrustedResolveGCS() {
		String qxri;

		try {
			Resolver resolver = setupResolver();
			qxri = "@blog*lockbox*peter2";
			XRI xri = XRI.fromIRINormalForm(qxri);

			ResolverFlags flags = new ResolverFlags();
			flags.setUric(true);
			flags.setHttps(false);
			flags.setSaml(true);

                        resolver.resolveSEPToXRD(xri, null, null, flags, new ResolverState());

			ArrayList uris = resolver.resolveSEPToURIList(qxri, new TrustType(), null, "text/xml", true);


		}
		catch (Exception e) {
			e.printStackTrace();
			fail("Not expecting an exception here: " + e);
		}
	}

        private static XRD createAuthRoot(String uri) {
            return createAuthRoot(uri, null);
        }

        private static XRD createAuthRoot(String uri, TrustType tt) {
		XRD xrd = new XRD();

		// construct an authority resolution service
		Service srv = new Service();
		if (tt == null)
                    tt = new TrustType(); // default trust type
                else {
                    try {
                        X509Certificate[] arr = new X509Certificate[1];
                        InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream("keystore.jks");
                        KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
                        ks.load(is, "password".toCharArray());
                        arr[0] = (X509Certificate)ks.getCertificate("issuercert");
                        Document oDoc = new DocumentImpl();
                        KeyInfo oKeyInfo = new KeyInfo(oDoc);
                        X509Data oData = new X509Data(oDoc);
                        oData.addCertificate(arr[0]);
                        oKeyInfo.add(oData);
                        srv.setKeyInfo(oKeyInfo);
                        
                        srv.setProviderId("@!E459.819D.771.7990!5B62.6F13.7602.5176");
                    } catch (Exception e) {
                        throw new RuntimeException("Failed to load keystore alias (certificate): " + e.toString());
                    }
                }
		String authMediaType = Tags.CONTENT_TYPE_XRDS + ";"
				+ tt.getParameterPair();
		srv.addMediaType(authMediaType, null, Boolean.FALSE);
		srv.addType(Tags.SERVICE_AUTH_RES);
		srv.addURI(uri);


		// add it to the XRD
		xrd.addService(srv);

		return xrd;
	}

	private static Resolver setupResolver()
	{
	  // instantiate a Resolver object
	  Resolver resolver = new Resolver();

	  // configure roots
	  XRD eqRoot = createAuthRoot("http://equal.xri.net/");
	  eqRoot.setCanonicalID(new CanonicalID("="));
	  Status eqRootStatus = new Status(Status.SUCCESS);
	  eqRootStatus.setCID(Status.CID_VERIFIED);
	  eqRoot.setStatus(eqRootStatus);
	  resolver.setAuthority("=", eqRoot);

	  XRD atRoot = createAuthRoot("http://at.xri.net/");
	  atRoot.setCanonicalID(new CanonicalID("@"));
	  Status atRootStatus = new Status(Status.SUCCESS);
	  atRootStatus.setCID(Status.CID_VERIFIED);
	  atRoot.setStatus(atRootStatus);
	  resolver.setAuthority("@", atRoot);

          TrustType tt = new TrustType();
          tt.setParameterPair(false, true);
	  XRD myRoot = createAuthRoot("http://localhost:8080/server/resolve/ns/@blog*lockbox/", tt);
	  log.debug("TrustedResolver() root hint  = " + myRoot.serializeDOM(false, true));
          myRoot.setCanonicalID(new CanonicalID("@!E459.819D.771.7990!5B62.6F13.7602.5176"));
	  Status myRootStatus = new Status(Status.SUCCESS);
	  myRootStatus.setCID(Status.CID_VERIFIED);
	  myRoot.setStatus(myRootStatus);
	  resolver.setAuthority("(@blog*lockbox)", myRoot); // note () form, whereas URL /ns/... does not
          return resolver;
	}
}
