package org.openxri.config.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.Properties;

import javax.servlet.ServletContext;

import org.openxri.config.ProxyConfig;
import org.openxri.xml.XRD;


/**
 * Provides an implementation of the ProxyConfig interface based on Java
 * properties.
 *
 * @author =chetan
 */
public class PropertiesProxyConfig implements ProxyConfig
{
	/**
	 * The proxy properties file path (relative to the current context root)
	 */
	public static final String PROXY_CONFIG_FILE = "proxy.config.file";
	public static final String DEFAULT_PROXY_CONFIG_FILE = "/WEB-INF/proxy.properties";

	private static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog( PropertiesProxyConfig.class.getName());

	protected Properties config;
	protected String configFile;

	public PropertiesProxyConfig(ServletContext servletContext, Properties properties) {

		this.config = null;

		// which file to read?

		this.configFile = properties.getProperty(PROXY_CONFIG_FILE, DEFAULT_PROXY_CONFIG_FILE);
		if (servletContext != null) this.configFile = servletContext.getRealPath(this.configFile);
	}

	public PropertiesProxyConfig(Properties config) {

		this.config = config;
		this.configFile = null;
	}

	public PropertiesProxyConfig(String configFile) {

		this.config = null;
		this.configFile = configFile;
	}

	public PropertiesProxyConfig() {

		this.config = null;
		this.configFile = null;
	}

	public void init() {

		// need to load config from file?

		if (this.config == null) {

			this.config = new Properties();

			// read from file

			InputStream stream;

			try {

				stream = new FileInputStream(new File(this.configFile));
				this.config.load(stream);
				stream.close();
			} catch (Exception ex) {

				PropertiesProxyConfig.logger.error("Cannot read properties file.", ex);
			}
		}
	}

	public void persist() {

		try {

			this.config.store(new FileOutputStream(this.configFile), null);
		} catch (Exception ex) {

			PropertiesProxyConfig.logger.error("Failed to store configuration properties", ex);
		}
	}

	public String getSetting(String key) 
	{
		return this.config.getProperty(key);
	}
	
	public String getProxyImplClass()
	{
		return this.config.getProperty(ProxyConfig.PROXY_CLASS, ProxyConfig.DEFAULT_PROXY_CLASS);

	}

	public void setProxyImplClass(String value)
	{
		this.setProp(ProxyConfig.PROXY_CLASS, value);

	}

	public XRD getEqualsAuthority()
	throws URISyntaxException, ParseException
	{
		String sDescriptor = this.config.getProperty("EqualsAuthority", null);
		return XRD.parseXRD(sDescriptor, false);

	}

	public void setEqualsAuthority(XRD oVal)
	{
		this.setProp("EqualsAuthority", oVal.toString());

	}

	public XRD getAtAuthority()
	throws URISyntaxException, ParseException
	{
		String sDescriptor = this.config.getProperty("AtAuthority", null);
		return XRD.parseXRD(sDescriptor, false);

	}

	public void setAtAuthority(XRD oVal)
	{
		this.setProp("AtAuthority", oVal.toString());

	}

	public XRD getBangAuthority()
	throws URISyntaxException, ParseException
	{
		String sDescriptor = this.config.getProperty("BangAuthority", null);
		return XRD.parseXRD(sDescriptor, false);

	}

	public void setBangAuthority(XRD oVal)
	{
		this.setProp("BangAuthority", oVal.toString());

	}

	public int getMaxBytesPerRequest() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_BYTES_PER_REQUEST, ProxyConfig.DEFAULT_MAX_BYTES_PER_REQUEST)));
	}

	public int getMaxFollowRedirects() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_FOLLOW_REDIRECTS, ProxyConfig.DEFAULT_MAX_FOLLOW_REDIRECTS)));
	}

	public int getMaxFollowRefs() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_FOLLOW_REFS, ProxyConfig.DEFAULT_MAX_FOLLOW_REFS)));
	}

	public int getMaxHttpRedirectAllowed() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_FOLLOW_REDIRECTS, ProxyConfig.DEFAULT_MAX_FOLLOW_REDIRECTS)));
	}

	public int getMaxRequests() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_REQUESTS, ProxyConfig.DEFAULT_MAX_REQUESTS)));
	}

	public int getMaxTotalBytes() {

		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_TOTAL_BYTES, ProxyConfig.DEFAULT_MAX_TOTAL_BYTES)));
	}

	public String[] getSupportedResMediaTypes() {

		String mediaTypes = this.config.getProperty(ProxyConfig.SUPPORTED_RES_MEDIA_TYPES, null);
		return(mediaTypes.split("\\s*,\\s*"));
	}

	public String[] getHttpsBypassAuthorities() {
		
		String authorities = this.config.getProperty(ProxyConfig.HTTPS_BYPASS_AUTHORITIES, null);
		if (authorities == null) {
			return new String[0];
		}
		return authorities.split("\\s+");
	}

	public String[] getSamlBypassAuthorities() {
		
		String authorities = this.config.getProperty(ProxyConfig.SAML_BYPASS_AUTHORITIES, null);
		if (authorities == null) {
			return new String[0];
		}
		return authorities.split("\\s+");
	}
	
	public void setMaxBytesPerRequest(int maxBytesPerRequest) {

		this.setProp(ProxyConfig.MAX_BYTES_PER_REQUEST, Integer.toString(maxBytesPerRequest));
	}

	public void setMaxFollowRedirects(int maxFollowRedirects) {

		this.setProp(ProxyConfig.MAX_FOLLOW_REDIRECTS, Integer.toString(maxFollowRedirects));
	}

	public void setMaxFollowRefs(int maxFollowRefs) {

		this.setProp(ProxyConfig.MAX_FOLLOW_REFS, Integer.toString(maxFollowRefs));
	}

	public void setMaxRequests(int maxRequests) {

		this.setProp(ProxyConfig.MAX_REQUESTS, Integer.toString(maxRequests));
	}

	public void setMaxTotalBytes(int maxTotalBytes) {

		this.setProp(ProxyConfig.MAX_TOTAL_BYTES, Integer.toString(maxTotalBytes));
	}

	public void setSupportedResMediaTypes(String[] mediaTypes) {

		StringBuffer join = new StringBuffer();;

		for (int i=0; i<mediaTypes.length; i++) join.append((i>0 ? ", " : "") + mediaTypes[i]);

		this.setProp(ProxyConfig.SUPPORTED_RES_MEDIA_TYPES, join.toString());
	}

	
	public void setHttpsBypassAuthorities(String[] authorities) {
		StringBuffer join = new StringBuffer();;
		for (int i = 0; i < authorities.length; i++) {
			if (join.length() > 0) join.append(" ");
			join.append(authorities[i]);
		}
		this.setProp(ProxyConfig.HTTPS_BYPASS_AUTHORITIES, join.toString());		
	}
	
	public void setSamlBypassAuthorities(String[] authorities) {
		StringBuffer join = new StringBuffer();;
		for (int i = 0; i < authorities.length; i++) {
			if (join.length() > 0) join.append(" ");
			join.append(authorities[i]);
		}
		this.setProp(ProxyConfig.SAML_BYPASS_AUTHORITIES, join.toString());		
	}

	public String getBareXRINotFoundRedirect() {
		return this.config.getProperty(ProxyConfig.BARE_XRI_NOTFOUND_REDIRECT);
	}
	
	public void setBareXRINotFoundRedirect(String bareXRINotFoundRedirect) {
		this.setProp(ProxyConfig.BARE_XRI_NOTFOUND_REDIRECT, bareXRINotFoundRedirect);
	}

	public String getRootRedirect() {
		return this.config.getProperty(ProxyConfig.ROOT_REDIRECT);
	}
	
	public void setRootRedirect(String rootRedirect) {
		this.setProp(ProxyConfig.ROOT_REDIRECT, rootRedirect);
	}
	
	public boolean getUseCache() {
		return Boolean.parseBoolean(this.config.getProperty(ProxyConfig.USE_CACHE, "true"));
	}
	
	public void setUseCache(boolean useCache) {
		this.setProp(ProxyConfig.USE_CACHE, Boolean.toString(useCache));
	}

	public int getDefaultCacheTTL() {
		return(Integer.parseInt(this.config.getProperty(ProxyConfig.DEFAULT_CACHE_TTL, ProxyConfig.DEFAULT_DEFAULT_CACHE_TTL)));
	}

	public void setDefaultCacheTTL(int defaultCacheTTL) {
		this.setProp(ProxyConfig.DEFAULT_CACHE_TTL, Integer.toString(defaultCacheTTL));
	}

	public int getMinCacheTTL() {
		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MIN_CACHE_TTL, ProxyConfig.DEFAULT_MIN_CACHE_TTL)));
	}

	public void setMinCacheTTL(int minCacheTTL) {
		this.setProp(ProxyConfig.MIN_CACHE_TTL, Integer.toString(minCacheTTL));
	}

	public int getMaxCacheTTL() {
		return(Integer.parseInt(this.config.getProperty(ProxyConfig.MAX_CACHE_TTL, ProxyConfig.DEFAULT_MAX_CACHE_TTL)));
	}

	public void setMaxCacheTTL(int maxCacheTTL) {
		this.setProp(ProxyConfig.MAX_CACHE_TTL, Integer.toString(maxCacheTTL));
	}

	public int getNegativeCacheTTL() {
		return(Integer.parseInt(this.config.getProperty(ProxyConfig.NEGATIVE_CACHE_TTL, ProxyConfig.DEFAULT_NEGATIVE_CACHE_TTL)));
	}

	public void setNegativeCacheTTL(int negativeCacheTTL) {
		this.setProp(ProxyConfig.NEGATIVE_CACHE_TTL, Integer.toString(negativeCacheTTL));
	}

	/**
	 * Sets a property
	 * @param sKey - The key to set
	 * @param sValue - The value to set the key to
	 */
	private void setProp(String sKey, String sValue)
	{
		if (sValue == null)
		{
			this.config.remove(sKey);
		}
		else
		{
			this.config.put(sKey, sValue);
		}
	}

}
