package org.openxri.config.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.openxri.config.Component;
import org.openxri.config.ComponentRegistry;
import org.openxri.config.impl.XMLServerConfig.ComponentConfig;

/**
 * Provides a single point to obtain fully configured Component instances.
 * Server components are: Server, Store, URIMapper, Plugin
 *
 * @author =peacekeeper
 */
class XMLComponentRegistry implements ComponentRegistry {

	private static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(XMLComponentRegistry.class.getName());

	private List<Component> componentList;
	private Map<Class<? extends Component>, Component> componentMap;

	XMLComponentRegistry(List<ComponentConfig> componentConfigs, ClassLoader classLoader) { 

		this.componentList = new ArrayList<Component> ();
		this.componentMap = new HashMap<Class<? extends Component>, Component> ();

		this.load(componentConfigs, classLoader);
	}

	XMLComponentRegistry(List<ComponentConfig> componentConfigs) { 

		this.componentList = new ArrayList<Component> ();
		this.componentMap = new HashMap<Class<? extends Component>, Component> ();

		this.load(componentConfigs, Thread.currentThread().getContextClassLoader());
	}

	@SuppressWarnings("unchecked")
	private void load(List<ComponentConfig> componentConfigs, ClassLoader classLoader) {

		// load components
		
		for (ComponentConfig componentConfig : componentConfigs) {

			// try to load component interface class

			String interfaceName = componentConfig.getInterfaceName();
			Class<? extends Component> interfaceClass;

			try {

				interfaceClass = (Class<? extends Component>) classLoader.loadClass(interfaceName);
			} catch (ClassNotFoundException ex) {

				log.error("Server component interface not found: " + interfaceName, ex);
				continue;
			}

			// try to load component implementation class

			String implementationName = componentConfig.getClassName();
			Class<? extends Component> implementationClass;

			try {

				implementationClass = (Class<? extends Component>) classLoader.loadClass(implementationName);
			} catch (ClassNotFoundException ex) {

				log.error("Server component implementation not found: " + implementationName, ex);
				continue;
			}

			// instantiate component

			Properties properties = componentConfig.getProperties();
			Component component;

			try {

				Class<?>[] constructorArgs = new Class<?>[] { Properties.class };
				Constructor<?> constructor = implementationClass.getConstructor(constructorArgs);
				component = (Component) constructor.newInstance(new Object[] { properties });
			} catch (NoSuchMethodException ex) {

				log.error("Server component " + implementationName + " does not have a constructor that takes a java.util.Properties object.", ex);
				continue;
			} catch (InstantiationException ex) {

				log.error("Cannot instantiate component " + implementationName + ".", ex);
				continue;
			} catch (IllegalAccessException ex) {

				log.error("Illegal access to constructor of component " + implementationName + ".", ex);
				continue;
			} catch (InvocationTargetException ex) {

				log.error("Cannot invoke constructor of component " + implementationName + ".", ex);
				continue;
			} catch (Exception ex) {

				log.error("Cannot construct component " + implementationName + ".", ex);
				continue;
			}

			// register it

			this.componentList.add(component);
			this.componentMap.put(interfaceClass, component);
		}
	}
	
	public void init() {
		
		// init components
		
		for (Component component : this.componentList) {
			
			try {
				
				component.init();
			} catch (Exception ex) {

				log.error("Cannot init component " + component.getClass().getName() + ".", ex);
				continue;
			}
		}
	}

	public List<Component> getComponents() {
		
		if (this.componentList == null) throw new IllegalStateException("Not initialized.");

		return(this.componentList);
	}

	public Component getComponent(Class<? extends Component> interfaceClass) {

		if (this.componentMap == null) throw new IllegalStateException("Not initialized.");

		return(this.componentMap.get(interfaceClass));
	}
}
