package org.openxri.config.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.openxri.config.PipelineRegistry;
import org.openxri.config.impl.XMLServerConfig.PipelineConfig;
import org.openxri.config.impl.XMLServerConfig.StageConfig;
import org.openxri.pipeline.Pipeline;
import org.openxri.pipeline.Stage;

/**
 * Provides a single point to obtain fully configured Pipeline instances.
 *
 * @author =peacekeeper
 */
class XMLPipelineRegistry implements PipelineRegistry {

	private static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(XMLPipelineRegistry.class.getName());

	public static final String DEFAULT_CREATE_PIPELINE_NAME = "create";
	public static final String DEFAULT_LOOKUP_PIPELINE_NAME = "lookup";

	private List<Pipeline> pipelineList;
	private Map<String, Pipeline> pipelineMap;

	XMLPipelineRegistry(List<PipelineConfig> pipelineConfigs, ClassLoader classLoader) { 

		this.load(pipelineConfigs, classLoader);
	}

	XMLPipelineRegistry(List<PipelineConfig> pipelineConfigs) { 

		this.load(pipelineConfigs, Thread.currentThread().getContextClassLoader());
	}

	@SuppressWarnings("unchecked")
	private void load(List<PipelineConfig> pipelineConfigs, ClassLoader classLoader) {

		this.pipelineList = new ArrayList<Pipeline> ();
		this.pipelineMap = new HashMap<String, Pipeline> ();

		for (PipelineConfig pipelineConfig : pipelineConfigs) {

			// try to load individual stages

			List<Stage> stages = new ArrayList<Stage> ();

			for (StageConfig stageConfig : pipelineConfig.getStageConfigs()) {

				// try to load stage implementation class

				String implementationName = stageConfig.getClassName();
				Class<? extends Stage> implementationClass;

				try {

					implementationClass = (Class<? extends Stage>) classLoader.loadClass(implementationName);
				} catch (ClassNotFoundException ex) {

					log.error("Stage implementation not found: " + implementationName, ex);
					continue;
				}

				// instantiate and configure stage

				Properties properties = stageConfig.getProperties();
				Stage stage;

				try {

					Class<?>[] constructorArgs = new Class<?>[] { Properties.class };
					Constructor<?> constructor = implementationClass.getConstructor(constructorArgs);
					stage = (Stage) constructor.newInstance(new Object[] { properties });
					stage.init();
				} catch (NoSuchMethodException ex) {

					log.error("Stage " + implementationName + " does not have a constructor that takes a java.util.Properties object.", ex);
					continue;
				} catch (InstantiationException ex) {

					log.error("Cannot instantiate stage " + implementationName + ".", ex);
					continue;
				} catch (IllegalAccessException ex) {

					log.error("Illegal access to constructor of stage " + implementationName + ".", ex);
					continue;
				} catch (InvocationTargetException ex) {

					log.error("Cannot invoke constructor of stage " + implementationName + ".", ex);
					continue;
				} catch (Exception ex) {

					log.error("Cannot construct stage " + implementationName + ".", ex);
					continue;
				}

				stages.add(stage);
			}

			// instantiate pipeline

			String name = pipelineConfig.getName();
			Pipeline pipeline = new Pipeline(name, stages);

			// register it

			this.pipelineList.add(pipeline);
			this.pipelineMap.put(name, pipeline);
		}
	}

	public void init() {
		
	}
	
	public List<Pipeline> getPipelines() {

		if (this.pipelineList == null) throw new IllegalStateException("Not initialized.");

		return(this.pipelineList);
	}

	public Pipeline getPipelineByName(String name) {

		if (this.pipelineMap == null) throw new IllegalStateException("Not initialized.");

		return(this.pipelineMap.get(name));
	}

	public Pipeline getDefaultCreatePipeline() {

		return(this.getPipelineByName(DEFAULT_CREATE_PIPELINE_NAME));
	}

	public Pipeline getDefaultLookupPipeline() {

		return(this.getPipelineByName(DEFAULT_LOOKUP_PIPELINE_NAME));
	}
}
