package org.openxri.config.impl;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.servlet.ServletContext;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.openxri.config.ComponentRegistry;
import org.openxri.config.PipelineRegistry;
import org.openxri.config.ServerConfig;
import org.openxri.exceptions.ConfigException;
import org.openxri.exceptions.ConfigParseException;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * This class parses the XML configuration file and then contains all information from that file.
 * 
 * It also holds an instance of ComponentRegistry and PipelineRegistry, which are fully initialized
 * once the ServerConfig object is constructed.
 * 
 * @author =peacekeeper
 */
public class XMLServerConfig extends org.xml.sax.helpers.DefaultHandler implements ServerConfig {

	public static final String SERVER_CONFIG_FILE = "server.config.file";
	public static final String DEFAULT_SERVER_CONFIG_FILE = "/WEB-INF/server.xml";

	public static final String NAMESPACE = "http://www.openxri.org/namespaces/serverconfig";
	public static final String ELEMENT_CLASS = "class";
	public static final String ELEMENT_PROPERTIES = "properties";
	public static final String ELEMENT_PROPERTY = "property";
	public static final String ELEMENT_COMPONENT = "component";
	public static final String ELEMENT_PIPELINE = "pipeline";
	public static final String ELEMENT_STAGE = "stage";
	public static final String ATTR_PROPERTY_KEY = "key";
	public static final String ATTR_PROPERTY_VALUE = "value";
	public static final String ATTR_COMPONENT_INTERFACE = "interface";
	public static final String ATTR_PIPELINE_NAME = "name";
	public static final String PIPELINE_DEFAULT_NAME = null;

	public static final String PROPERTIES_KEY_HOSTNAME = "hostname";
	public static final String PROPERTIES_KEY_HOSTPORT = "hostport";
	public static final String PROPERTIES_KEY_SERVLETPATH = "servletpath";

	private ServletContext servletContext;
	private String filename;

	private Properties properties;
	private XMLComponentRegistry componentRegistry;
	private XMLPipelineRegistry pipelineRegistry;

	private List<ComponentConfig> componentConfigs;
	private List<PipelineConfig> pipelineConfigs;

	private Delegate delegate;
	private SAXParseException ex;

	public XMLServerConfig(ServletContext servletContext, Properties properties) {

		this.servletContext = servletContext;
		
		this.properties = null;
		this.componentConfigs = new ArrayList<ComponentConfig> ();
		this.pipelineConfigs = new ArrayList<PipelineConfig> ();

		// which file to read?

		this.filename = properties.getProperty(SERVER_CONFIG_FILE, DEFAULT_SERVER_CONFIG_FILE);
		if (servletContext != null) this.filename = servletContext.getRealPath(this.filename);
	}

	public void init() throws IOException, ConfigException {

		// init from file

		InputSource source;
		File file = new File(this.filename);
		
		if (file.exists()) {
			
			source = new InputSource(new FileReader(file));
		} else {
			
			source = new InputSource(this.getClass().getResourceAsStream(this.filename));
		}
		
		this.parse(source);

		this.componentRegistry = new XMLComponentRegistry(this.componentConfigs);
		this.pipelineRegistry = new XMLPipelineRegistry(this.pipelineConfigs);
		
		this.componentRegistry.init();
		this.pipelineRegistry.init();
	}

	public Properties getProperties() {

		return(this.properties);
	}

	public ComponentRegistry getComponentRegistry() {

		return(this.componentRegistry);
	}

	public PipelineRegistry getPipelineRegistry() {

		return(this.pipelineRegistry);
	}

	public String getHostName() {

		return(this.properties.getProperty(PROPERTIES_KEY_HOSTNAME));
	}

	public String getHostPort() {

		return(this.properties.getProperty(PROPERTIES_KEY_HOSTPORT));
	}

	public String getServletPath() {

		return(this.properties.getProperty(PROPERTIES_KEY_SERVLETPATH));
	}

	public String getRealPath(String relativePath) {

		return this.servletContext.getRealPath(relativePath);
	}

	protected void parse(org.xml.sax.InputSource input) throws IOException, ConfigException {

		SAXParserFactory parserFactory;
		SAXParser parser;

		// read XML document

		parserFactory = SAXParserFactory.newInstance();
		parserFactory.setNamespaceAware(true);

		try {

			parser = parserFactory.newSAXParser();
		} catch (ParserConfigurationException ex) {

			throw new ConfigParseException(0, "Cannot configure XML parser.", ex);
		} catch (SAXException ex) {

			throw new ConfigParseException(0, "Cannot create XML parser.", ex);
		}

		// parse it

		try {

			parser.parse(input, this);
		} catch (SAXException ex) {

			int line = 0;
			if (this.ex != null) {

				line = ((SAXParseException) ex).getLineNumber();
				ex = this.ex;
			}

			throw new ConfigParseException(line, "Error parsing XML graph: " + ex.getMessage(), ex);
		}
	}

	@Override
	public void startDocument() throws SAXException {

		this.delegate = null;
	}

	@Override
	public void endDocument() throws SAXException {

		this.delegate = null;
	}

	@Override
	public void characters(char[] ch, int start, int length) throws SAXException {

		if (this.delegate != null) {

			this.delegate.characters(ch, start, length);
			return;
		}
	}

	@Override
	public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

		if (this.delegate != null) {

			this.delegate.startElement(namespaceURI, localName, qName, atts);
			return;
		}

		if (! NAMESPACE.equals(namespaceURI)) return;

		// <properties> element ?

		if (ELEMENT_PROPERTIES.equals(localName)) {

			// delegate to PropertiesConfig

			this.delegate = new PropertiesConfig(namespaceURI, localName, qName, atts);
			return;
		}

		// <component> element ?

		if (ELEMENT_COMPONENT.equals(localName)) {

			// delegate to ComponentConfig

			this.delegate = new ComponentConfig(namespaceURI, localName, qName, atts);
			return;
		}

		// <pipeline> element ?

		if (ELEMENT_PIPELINE.equals(localName)) {

			// delegate to PipelineConfig

			this.delegate = new PipelineConfig(namespaceURI, localName, qName, atts);
			return;
		}
	}

	@Override
	public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

		if (this.delegate != null && this.delegate.delegate != null) {

			this.delegate.endElement(namespaceURI, localName, qName);
			return;
		}

		if (! NAMESPACE.equals(namespaceURI)) return;

		// <properties> element ?

		if (ELEMENT_PROPERTIES.equals(localName)) {

			// end PropertiesConfig

			XMLServerConfig.this.properties = ((PropertiesConfig) this.delegate).getProperties();
			this.delegate = null;
			return;
		}

		// <component> element ?

		if (ELEMENT_COMPONENT.equals(localName)) {

			// end ComponentConfig

			if (((ComponentConfig) this.delegate).getInterfaceName() == null) return;
			XMLServerConfig.this.componentConfigs.add((ComponentConfig) this.delegate);
			this.delegate = null;
			return;
		}

		// <pipeline> element ?

		if (ELEMENT_PIPELINE.equals(localName)) {

			// end PipelineConfig

			if (((PipelineConfig) this.delegate).getName() == null) return;
			XMLServerConfig.this.pipelineConfigs.add((PipelineConfig) this.delegate);
			this.delegate = null;
			return;
		}
	}

	@Override
	public void error(SAXParseException ex) throws SAXException {

		this.ex = ex;
	}

	@Override
	public void fatalError(SAXParseException ex) throws SAXException {

		this.ex = ex;
	}

	@Override
	public void warning(SAXParseException ex) throws SAXException {

		this.ex = ex;
	}

	static abstract class Delegate extends DefaultHandler {
		
		protected Delegate delegate;
	}

	/**
	 * A SAX ContentHandler that can read a <pipeline> element.
	 * 
	 * @author =peacekeeper
	 */
	static class PipelineConfig extends Delegate {

		private String name;
		private List<StageConfig> stages;

		protected PipelineConfig(String namespaceURI, String localName, String qName, Attributes atts) {

			try {

				this.name = atts.getValue(XMLServerConfig.ATTR_PIPELINE_NAME);
			} catch (Exception ex) {

				this.name = XMLServerConfig.PIPELINE_DEFAULT_NAME;
			}

			this.stages = new ArrayList<StageConfig> ();
		}

		public String getName() {

			return(this.name);
		}

		public List<StageConfig> getStageConfigs() {

			return(this.stages);
		}

		@Override
		public void characters(char[] ch, int start, int length) throws SAXException {

			if (this.delegate != null) {

				this.delegate.characters(ch, start, length);
				return;
			}
		}

		@Override
		public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

			if (this.delegate != null) {

				this.delegate.startElement(namespaceURI, localName, qName, atts);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <stage> element ?

			if (XMLServerConfig.ELEMENT_STAGE.equals(localName)) {

				// delegate to StageConfig

				this.delegate = new StageConfig(namespaceURI, localName, qName, atts);
				return;
			}
		}

		@Override
		public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

			if (this.delegate != null && this.delegate.delegate != null) {

				this.delegate.endElement(namespaceURI, localName, qName);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <stage> element ?

			if (XMLServerConfig.ELEMENT_STAGE.equals(localName)) {

				// end StageConfig

				this.stages.add((StageConfig) this.delegate);
				this.delegate = null;
				return;
			}
		}
	}

	/**
	 * A SAX ContentHandler that can read a <component> element.
	 * 
	 * @author =peacekeeper
	 */
	static class ComponentConfig extends Delegate {

		private String interfaceName;
		private String className;
		private Properties properties;

		protected ComponentConfig(String namespaceURI, String localName, String qName, Attributes atts) {

			this.interfaceName = atts.getValue(XMLServerConfig.ATTR_COMPONENT_INTERFACE);
		}

		public String getInterfaceName() {

			return(this.interfaceName);
		}

		public String getClassName() {

			return(this.className);
		}

		public Properties getProperties() {

			return(this.properties);
		}

		@Override
		public void characters(char[] ch, int start, int length) throws SAXException {

			if (this.delegate != null) {

				this.delegate.characters(ch, start, length);
				return;
			}
		}

		@Override
		public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

			if (this.delegate != null) {

				this.delegate.startElement(namespaceURI, localName, qName, atts);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <class> element ?

			if (XMLServerConfig.ELEMENT_CLASS.equals(localName)) {

				// delegate to StringHandler

				this.delegate = new StringConfig(namespaceURI, localName, qName, atts);
				return;
			}

			// <properties> element ?

			if (XMLServerConfig.ELEMENT_PROPERTIES.equals(localName)) {

				// delegate to PropertiesHandler

				this.delegate = new PropertiesConfig(namespaceURI, localName, qName, atts);
				return;
			}
		}

		@Override
		public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

			if (this.delegate != null && this.delegate.delegate != null) {

				this.delegate.endElement(namespaceURI, localName, qName);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <class> element ?

			if (XMLServerConfig.ELEMENT_CLASS.equals(localName)) {

				// end StringHandler

				this.className = ((StringConfig) this.delegate).getString();
				this.delegate = null;
				return;
			}

			// <properties> element ?

			if (XMLServerConfig.ELEMENT_PROPERTIES.equals(localName)) {

				// end PropertiesHandler

				this.properties = ((PropertiesConfig) this.delegate).getProperties();
				this.delegate = null;
				return;
			}
		}
	}

	/**
	 * A SAX ContentHandler that can read any element that just contains a string.
	 * 
	 * @author =peacekeeper
	 */
	static class StringConfig extends Delegate {

		private String string;

		protected StringConfig(String namespaceURI, String localName, String qName, Attributes atts) {

			this.string = "";
		}

		public String getString() {

			return(this.string);
		}

		@Override
		public void characters(char[] ch, int start, int length) throws SAXException {

			char[] slice = new char[length];
			String str;

			System.arraycopy(ch, start, slice, 0, length);
			str = new String(slice);

			this.string += str;
		}
	}
	/**
	 * A SAX ContentHandler that can read a <stage> element.
	 * 
	 * @author =peacekeeper
	 */
	static class StageConfig extends Delegate {

		private String className;
		private Properties properties;

		protected StageConfig(String namespaceURI, String localName, String qName, Attributes atts) {

		}

		public String getClassName() {

			return(this.className);
		}

		public Properties getProperties() {

			return(this.properties);
		}

		@Override
		public void characters(char[] ch, int start, int length) throws SAXException {

			if (this.delegate != null) {

				this.delegate.characters(ch, start, length);
				return;
			}
		}

		@Override
		public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

			if (this.delegate != null) {

				this.delegate.startElement(namespaceURI, localName, qName, atts);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <class> element ?

			if (XMLServerConfig.ELEMENT_CLASS.equals(localName)) {

				// delegate to StringConfig

				this.delegate = new StringConfig(namespaceURI, localName, qName, atts);
				return;
			}

			// <properties> element ?

			if (XMLServerConfig.ELEMENT_PROPERTIES.equals(localName)) {

				// delegate to PropertiesConfig

				this.delegate = new PropertiesConfig(namespaceURI, localName, qName, atts);
				return;
			}
		}

		@Override
		public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

			if (this.delegate != null && this.delegate.delegate != null) {

				this.delegate.endElement(namespaceURI, localName, qName);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <class> element ?

			if (XMLServerConfig.ELEMENT_CLASS.equals(localName)) {

				// end StringConfig

				this.className = ((StringConfig) this.delegate).getString();
				this.delegate = null;
				return;
			}

			// <properties> element ?

			if (XMLServerConfig.ELEMENT_PROPERTIES.equals(localName)) {

				// end PropertiesConfig

				this.properties = ((PropertiesConfig) this.delegate).getProperties();
				this.delegate = null;
				return;
			}
		}
	}

	/**
	 * A SAX ContentHandler that can read a <properties> element.
	 * 
	 * @author =peacekeeper
	 */
	static class PropertiesConfig extends Delegate {

		private Properties properties;

		protected PropertiesConfig(String namespaceURI, String localName, String qName, Attributes atts) {

			this.properties = new Properties();
		}

		public Properties getProperties() {

			return(this.properties);
		}

		@Override
		public void characters(char[] ch, int start, int length) throws SAXException {

			if (this.delegate != null) {

				this.delegate.characters(ch, start, length);
				return;
			}
		}

		@Override
		public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;

			// <property> element ?

			if (XMLServerConfig.ELEMENT_PROPERTY.equals(localName)) {

				// read attributes

				String key = atts.getValue(XMLServerConfig.ATTR_PROPERTY_KEY);
				String value = atts.getValue(XMLServerConfig.ATTR_PROPERTY_VALUE);

				if (key == null) return;

				// store it

				this.properties.setProperty(key, value);
			}
		}

		@Override
		public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

			if (this.delegate != null && this.delegate.delegate != null) {

				this.delegate.endElement(namespaceURI, localName, qName);
				return;
			}

			if (! XMLServerConfig.NAMESPACE.equals(namespaceURI)) return;
		}
	}
}
