package org.openxri.factories;

import java.lang.reflect.Constructor;
import java.util.Enumeration;
import java.util.Properties;

import javax.servlet.FilterConfig;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

import org.openxri.config.ProxyConfig;
import org.openxri.config.impl.PropertiesProxyConfig;

/**
 * Can construct an appropriate ProxyConfig object based on a map (which at a minimum must 
 * contain a class name for the ProxyConfig implementation). 
 * 
 * @author =peacekeeper
 */
public class ProxyConfigFactory {

	private static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(ProxyConfigFactory.class.getName());

	public static final String PROXY_CONFIG_CLASS = "proxy.config.class";
	public static final String DEFAULT_PROXY_CONFIG_CLASS = PropertiesProxyConfig.class.getName();

	private static ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
	private static ProxyConfig instance = null;

	private ProxyConfigFactory() { }

	/**
	 * Construct the concrete implementation of the ProxyConfig
	 * interface, based on a servlet context and properties.
	 */
	public static synchronized ProxyConfig initSingleton(ServletContext servletContext, Properties properties) throws Exception {

		String className = (String) properties.getProperty(PROXY_CONFIG_CLASS);
		if (className == null) className = DEFAULT_PROXY_CONFIG_CLASS;

		log.info("Using ProxyConfig implementation class: " + className + ".");

		// find the class, invoke the constructor with a ServletConfig argument, then call the init() method.

		Class<?> clazz = classLoader.loadClass(className);

		Constructor<?> constructor = clazz.getConstructor(ServletContext.class, Properties.class);
		instance = (ProxyConfig) constructor.newInstance(servletContext, properties);

		instance.init();

		// done

		log.debug("Successfully constructed ProxyConfig implementation: " + className);
		return(instance);
	} 

	/**
	 * Construct the concrete implementation of the ProxyConfig
	 * interface, based on a servlet config.
	 */
	public static synchronized ProxyConfig initSingleton(ServletConfig servletConfig) throws Exception {

		Properties properties = new Properties();
		
		for (Enumeration e = servletConfig.getInitParameterNames(); e.hasMoreElements(); ) {
			
			String initParameterName = (String) e.nextElement();
			
			properties.setProperty(initParameterName, servletConfig.getInitParameter(initParameterName));
		}

		return(initSingleton(servletConfig.getServletContext(), properties));
	} 

	/**
	 * Construct the concrete implementation of the ProxyConfig
	 * interface, based on a filter config.
	 */
	public static synchronized ProxyConfig initSingleton(FilterConfig filterConfig) throws Exception {

		Properties properties = new Properties();
		
		for (Enumeration e = filterConfig.getInitParameterNames(); e.hasMoreElements(); ) {
			
			String initParameterName = (String) e.nextElement();
			
			properties.setProperty(initParameterName, filterConfig.getInitParameter(initParameterName));
		}

		return(initSingleton(filterConfig.getServletContext(), properties));
	} 

	/**
	 * Get the proxy config implementation and hope it's initialized already.
	 */
	public static synchronized ProxyConfig getSingleton() {

		return(instance);
	}

	public static synchronized ClassLoader getClassLoader() {

		return(ProxyConfigFactory.classLoader);
	}

	public static synchronized void setClassLoader(ClassLoader classLoader) {

		ProxyConfigFactory.classLoader = classLoader;
	}
}
