package org.openxri.factories;

import java.lang.reflect.Constructor;
import java.util.Enumeration;
import java.util.Properties;

import javax.servlet.FilterConfig;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

import org.openxri.config.ServerConfig;
import org.openxri.config.impl.XMLServerConfig;

/**
 * Can construct an appropriate ServerConfig object based on a map (which at a minimum must 
 * contain a class name for the ServerConfig implementation). 
 * 
 * @author =peacekeeper
 */
public class ServerConfigFactory {

	private static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(ServerConfigFactory.class.getName());

	public static final String SERVER_CONFIG_CLASS = "server.config.class";
	public static final String DEFAULT_SERVER_CONFIG_CLASS = XMLServerConfig.class.getName();

	private static ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
	private static ServerConfig instance = null;

	private ServerConfigFactory() { }

	/**
	 * Construct the concrete implementation of the ServerConfig
	 * interface, based on a servlet context and properties.
	 */
	public static synchronized ServerConfig initSingleton(ServletContext servletContext, Properties properties) throws Exception {

		String className = properties.getProperty(SERVER_CONFIG_CLASS);
		if (className == null) className = DEFAULT_SERVER_CONFIG_CLASS;

		log.info("Using ServerConfig implementation class: " + className + ".");

		// find the class, invoke the constructor with a ServletContext and a Properties argument, then call the init() method.

		Class<?> clazz = classLoader.loadClass(className);

		Constructor<?> constructor = clazz.getConstructor(ServletContext.class, Properties.class);
		instance = (ServerConfig) constructor.newInstance(servletContext, properties);

		instance.init();

		// done

		log.debug("Successfully constructed ServerConfig implementation: " + className);
		return(instance);
	} 

	/**
	 * Construct the concrete implementation of the ServerConfig
	 * interface, based on a servlet config.
	 */
	public static synchronized ServerConfig initSingleton(ServletConfig servletConfig) throws Exception {

		Properties properties = new Properties();

		for (Enumeration e = servletConfig.getInitParameterNames(); e.hasMoreElements(); ) {

			String initParameterName = (String) e.nextElement();

			properties.setProperty(initParameterName, servletConfig.getInitParameter(initParameterName));
		}

		return(initSingleton(servletConfig.getServletContext(), properties));
	} 

	/**
	 * Construct the concrete implementation of the ServerConfig
	 * interface, based on a filter config.
	 */
	public static synchronized ServerConfig initSingleton(FilterConfig filterConfig) throws Exception {

		Properties properties = new Properties();

		for (Enumeration e = filterConfig.getInitParameterNames(); e.hasMoreElements(); ) {

			String initParameterName = (String) e.nextElement();

			properties.setProperty(initParameterName, filterConfig.getInitParameter(initParameterName));
		}

		return(initSingleton(filterConfig.getServletContext(), properties));
	} 

	/**
	 * Get the server config implementation and hope it's initialized already.
	 */
	public static synchronized ServerConfig getSingleton() {

		return(instance);
	}

	public static synchronized ClassLoader getClassLoader() {

		return(ServerConfigFactory.classLoader);
	}

	public static synchronized void setClassLoader(ClassLoader classLoader) {

		ServerConfigFactory.classLoader = classLoader;
	}
}
