package org.openxri.pipeline;

import java.io.Serializable;
import java.util.List;

import org.openxri.XRISegment;
import org.openxri.exceptions.StageException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.xml.XRD;

/**
 * A pipeline's job is to assemble an XRD. This is done by executing a number of stages which can each contribute to the XRD.
 * 
 * @author =peacekeeper
 */
public class Pipeline implements Serializable {

	/**
	 * Authorities can override the CREATE pipeline by setting the value of this attribute to the name of
	 * one of the pipelines in the server configuration. This means that the overridden pipeline is executed
	 * instead of the default one whenever a new authority is created under this authority.
	 */
	public static final String ATTRIBUTE_OVERRIDE_CREATE_PIPELINE = "override-create-pipeline";

	/**
	 * Authorities can override the LOOKUP pipeline by setting the value of this attribute to the name of
	 * one of the pipelines in the server configuration. This means that the overridden pipeline is executed
	 * instead of the default one whenever this authority is resolved.
	 */
	public static final String ATTRIBUTE_OVERRIDE_LOOKUP_PIPELINE = "override-lookup-pipeline";

	private String name;
	private List<Stage> stages;

	public Pipeline(String name, List<Stage> stages) {

		this.name = name;
		this.stages = stages;
	}

	public String getName() {

		return(this.name);
	}

	public Stage[] getStages() {
		
		return(this.stages.toArray(new Stage[this.stages.size()]));
	}
	
	/**
	 * A pipeline is executed by consecutively executing all its stages. The result XRD of one stage becomes the input XRD
	 * of the next.
	 * 
	 * @param store The store from which the authorities were retrieved.
	 * @param xrd A XRD that is passed into the pipeline.
	 * @param segment The XRI segment that resolves to the XRD being processed.
	 * @param parent The parent authority, i.e. the authority describing the XRD.
	 * @param subSegmentName The name of the subsegment that connects the two authorities.
	 * @param authority The authority, i.e. the authority described by the XRD.
	 * @return The result XRD of the pipeline.
	 * @throws StageException
	 */
	public XRD execute(Store store, XRD xrd, XRISegment segment, Authority parentAuthority, String subSegmentName, Authority authority, boolean isCreate) throws StageException {

		for (Stage stage : this.stages) {

			xrd = stage.execute(store, xrd, segment, parentAuthority, subSegmentName, authority, isCreate);

			if (xrd == null) return(null);
		}

		return(xrd);
	}
}
