package org.openxri.pipeline.stages;

import java.util.Properties;
import java.util.Random;

import org.openxri.XRISegment;
import org.openxri.exceptions.StageException;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.xml.CanonicalID;
import org.openxri.xml.XRD;

/**
 * A stage that adds a <CanonicalID> element to the XRD.
 * The CanonicalID is constructed from the parent authority's CanonicalID plus a random persistent 
 * subsegment of the form of a GUID, e.g. !xxxx.xxxx.xxxx.xxxx.xxxx.xxxx.xxxx.xxxx
 * 
 * Parameters for the stage's configuration:
 * (none)
 * 
 * Recommended pipeline(s) for this stage: 
 * CREATE
 * 
 * @author =peacekeeper
 */
public class AddGUIDCanonicalIDStage extends AbstractStage {

	private static final Random random = new Random();
	private static final char[] hexchars = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };  

	public AddGUIDCanonicalIDStage(Properties properties) {

		super(properties);
	}

	public void init() throws Exception {

	}

	public XRD execute(Store store, XRD xrd, XRISegment segment, Authority parentAuthority, String subSegmentName, Authority authority, boolean isCreate) throws StageException {

		// can only create a CanonicalID if the parent authority has one

		if (parentAuthority == null) return(xrd);
		if (parentAuthority.getXrd().getNumCanonicalids() < 1) return(xrd);

		// special case: if the new subsegment already is an i-number, we don't generate anything
		
		boolean isINumber = false;
		
		if (subSegmentName.startsWith("!")) isINumber = true;
		
		// create a new CanonicalID (random)

		String localCanonicalIDString = isINumber ? subSegmentName : randomCanonicalIDString();
		String canonicalIDString = parentAuthority.getXrd().getCanonicalID().getValue() + localCanonicalIDString;
		CanonicalID canonicalID = new CanonicalID(canonicalIDString);

		// put the CanonicalID into the XRD

		xrd.setCanonicalID(canonicalID);

		// if this is a CREATE pipeline, put the new CanonicalID in the store as a subsegment, so it can be resolved properly

		if (isCreate && ! isINumber) {

			try {

				store.registerSubsegment(parentAuthority, localCanonicalIDString, authority);
			} catch (StoreException ex) {

				throw new StageException("Cannot register subsegment for CanonicalID.");
			}
		}

		// done

		return(xrd);
	}

	public static String randomCanonicalIDString() {

		String str = "!";

		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();
		str += '.';
		str += fourRandomChars();

		return(str);
	}

	private static String fourRandomChars() {

		return("" + randomHexChar() + randomHexChar() + randomHexChar() + randomHexChar());
	}

	private static char randomHexChar() {

		return(hexchars[random.nextInt(hexchars.length)]);
	}
}
