package org.openxri.pipeline.stages;

import java.util.Map;
import java.util.Properties;

import org.openxri.XRISegment;
import org.openxri.exceptions.StageException;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.xml.CanonicalID;
import org.openxri.xml.XRD;

/**
 * A stage that adds a <CanonicalID> element to the XRD.
 * The CanonicalID is constructed from the parent authority's CanonicalID plus incremental persistent subsegments (!1, !2, !3, ...)
 * 
 * Parameters for the stage's configuration:
 * (none)
 * 
 * Recommended pipeline(s) for this stage: 
 * CREATE
 * 
 * @author =peacekeeper
 */
public class AddSerialCanonicalIDStage extends AbstractStage {

	public static final String ATTRIBUTES_KEY_LAST_CANONICALID = "last-canonicalid";

	public AddSerialCanonicalIDStage(Properties properties) {

		super(properties);
	}

	public void init() throws Exception {

	}

	public XRD execute(Store store, XRD xrd, XRISegment segment, Authority parentAuthority, String subSegmentName, Authority authority, boolean isCreate) throws StageException {

		if (! (store instanceof StoreAttributable)) throw new StageException("Cannot use this store implementation.");
		StoreAttributable storeAttributable = (StoreAttributable) store;

		// can only create a CanonicalID if the parent authority has one

		if (parentAuthority == null) return(xrd);
		if (parentAuthority.getXrd().getNumCanonicalids() < 1) return(xrd);

		// special case: if the new subsegment already is an i-number, we don't generate anything
		
		boolean isINumber = false;
		
		if (subSegmentName.startsWith("!")) isINumber = true;

		// create a new CanonicalID (read from authority attribute, increment by 1, write back as authority attribute)

		String canonicalIDString = null;
		int canonicalIDValue = -1;

		if (! isINumber) {

			try {
	
				canonicalIDString = storeAttributable.getAuthorityAttributes(parentAuthority).get(ATTRIBUTES_KEY_LAST_CANONICALID);
				canonicalIDValue = Integer.parseInt(canonicalIDString);
				canonicalIDValue++;
			} catch (StoreException ex) {
	
				throw new StageException("Cannot read authority attribute from store.", ex);
			} catch (NullPointerException ex) {		// if the attribute does not yet exist on that authority
	
				canonicalIDValue = 0;
			} catch (NumberFormatException ex) {	// if the attribute did not contain a valid integer value
	
				canonicalIDValue = 0;
			}
		}

		String localCanonicalIDString = isINumber ? subSegmentName : ("!" + Integer.toString(canonicalIDValue));
		canonicalIDString = parentAuthority.getXrd().getCanonicalidAt(0).getValue() + localCanonicalIDString;

		CanonicalID canonicalID = new CanonicalID(canonicalIDString);

		// write incremented authority attribute back to store
		
		if (! isINumber) {
		
			try {
	
				Map<String, String> attributes = storeAttributable.getAuthorityAttributes(parentAuthority);
				attributes.put(ATTRIBUTES_KEY_LAST_CANONICALID, Integer.toString(canonicalIDValue));
	
				storeAttributable.setAuthorityAttributes(parentAuthority, attributes);
			} catch (StoreException ex) {
	
				throw new StageException("Cannot write authority attribute to store.", ex);
			}
		}

		// put the CanonicalID into the XRD

		xrd.addCanonicalID(canonicalID);

		// if this is a CREATE pipeline, put the new CanonicalID into the store as a subsegment, so it can be resolved properly

		if (isCreate && ! isINumber) {

			try {

				store.registerSubsegment(parentAuthority, localCanonicalIDString, authority);
			} catch (StoreException ex) {

				throw new StageException("Cannot register subsegment for CanonicalID.");
			}
		}

		// done

		return(xrd);
	}
}
