package org.openxri.pipeline.stages;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.openxri.XRISegment;
import org.openxri.exceptions.StageException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.xml.XRD;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * A stage that adds an arbitrary XML element to the XRD.
 * The text value of the element and the element's attributes can be configured using stage parameters.
 * 
 * Parameters for the stage's configuration:
 * 
 * elementname (required):
 * The name of the element.
 * 
 * elementnamespace (optional, default: none):
 * The namespace of the element.
 * 
 * elementvalue (optional, default: empty string):
 * The text inside the element.
 * 
 * [other parameters] (optional):
 * Other parameters are treated as attributes for the XML element. The parameter name is the
 * attribute name, and the parameter value is the attribute value.
 * 
 * Recommended pipeline(s) for this stage: 
 * CREATE, LOOKUP
 * 
 * @author =peacekeeper
 */
public class AddXMLElementStage extends AbstractStage {

	public static final String PROPERTIES_KEY_ELEMENTNAME = "elementname";
	public static final String PROPERTIES_KEY_ELEMENTNAMESPACE = "elementnamespace";
	public static final String PROPERTIES_KEY_ELEMENTVALUE = "elementvalue";

	public static final String DEFAULT_ELEMENTNAMESPACE = null;
	public static final String DEFAULT_ELEMENTVALUE = "";

	private String elementName;
	private String elementNamespace;
	private String elementValue;
	private String[] attributeNames;
	private String[] attributeValues;

	public AddXMLElementStage(Properties properties) {

		super(properties);
	}

	public void init() throws Exception {

		String elementName = this.properties.getProperty(PROPERTIES_KEY_ELEMENTNAME);
		if (elementName == null) throw new StageException("Required property " + PROPERTIES_KEY_ELEMENTNAME + " not found.");
		this.elementName = elementName;

		String elementNamespace = this.properties.getProperty(PROPERTIES_KEY_ELEMENTNAMESPACE, DEFAULT_ELEMENTNAMESPACE);
		this.elementNamespace = elementNamespace;

		String elementValue = this.properties.getProperty(PROPERTIES_KEY_ELEMENTVALUE, DEFAULT_ELEMENTVALUE);
		this.elementValue = elementValue;

		List<String> attributeNames = new ArrayList<String> ();
		List<String> attributeValues = new ArrayList<String> ();

		for (Object key : this.properties.keySet()) {

			String attributeName = (String) key;
			String attributeValue = this.properties.getProperty(attributeName);

			if (attributeName.equals(PROPERTIES_KEY_ELEMENTNAME) || 
					attributeName.equals(PROPERTIES_KEY_ELEMENTNAMESPACE) || 
					attributeName.equals(PROPERTIES_KEY_ELEMENTVALUE) || 
					attributeValue == null) continue;

			attributeNames.add(attributeName);
			attributeValues.add(attributeValue);
		}

		this.attributeNames = attributeNames.toArray(new String[attributeNames.size()]);
		this.attributeValues = attributeValues.toArray(new String[attributeValues.size()]);
	}

	public XRD execute(Store store, XRD xrd, XRISegment segment, Authority parentAuthority, String subSegmentName, Authority authority, boolean isCreate) throws StageException {

		if (! (store instanceof StoreAttributable)) throw new StageException("Cannot use this store implementation.");
		StoreAttributable storeAttributable = (StoreAttributable) store;

		Element xrdElement = xrd.getDOM();
		Document document = xrdElement.getOwnerDocument();

		// create new element

		Element element;

		if (this.elementNamespace != null) {

			element = document.createElementNS(this.elementNamespace, this.elementName);
		} else {

			element = document.createElement(this.elementName);
		}

		if (this.elementValue != null) {

			Text text = document.createTextNode(this.elementValue);
			element.appendChild(text);
		}

		for (int i=0; i<this.attributeNames.length; i++) {

			Attr attr = document.createAttribute(this.attributeNames[i]);
			attr.setNodeValue(this.attributeValues[i]);
			element.setAttributeNode(attr);
		}

		// insert element into XRD

		try {

			xrdElement.appendChild(element);
			xrd = new XRD(xrdElement, false);
		} catch (Exception ex) {

			throw new StageException("Unable to parse modified XRD.", ex);
		}

		// done

		return(xrd);
	}
}
