package org.openxri.pipeline.stages;

import java.util.Properties;

import org.openxri.XRISegment;
import org.openxri.exceptions.StageException;
import org.openxri.exceptions.StoreException;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreBetterLookup;
import org.openxri.store.SubSegment;
import org.openxri.xml.LocalID;
import org.openxri.xml.XRD;

/**
 * A stage that automatically generates <LocalID> elements by finding synonyms in the store.
 * 
 * Parameters for the stage's configuration:
 * 
 * excludeself (optional, default: true):
 * True, if no <LocalID> should be created for the subsegment that is being looked up. 
 * 
 * Recommended pipeline(s) for this stage: 
 * LOOKUP
 * 
 * @author =peacekeeper
 */
public class AutoLocalIDsStage extends AbstractStage {

	public static final String PROPERTIES_KEY_EXCLUDESELF = "excludeself";

	public static final String DEFAULT_EXCLUDESELF = "true";

	private boolean excludeSelf;

	public AutoLocalIDsStage(Properties properties) {

		super(properties);
	}

	public void init() throws Exception {

		String excludeSelf = this.properties.getProperty(PROPERTIES_KEY_EXCLUDESELF, DEFAULT_EXCLUDESELF);
		this.excludeSelf = Boolean.parseBoolean(excludeSelf);
	}

	public XRD execute(Store store, XRD xrd, XRISegment segment, Authority parentAuthority, String subSegmentName, Authority authority, boolean isCreate) throws StageException {

		if (! (store instanceof StoreBetterLookup)) throw new StageException("Cannot use this store implementation.");
		StoreBetterLookup storeBetterLookup = (StoreBetterLookup) store;

		if (parentAuthority == null) return(xrd);

		// get all subsegments with the given parent authority and authority

		SubSegment[] localSubSegments;

		try {

			localSubSegments = storeBetterLookup.getSynonymSubSegments(parentAuthority, authority);
		} catch (StoreException ex) {

			throw new StageException("Cannot access store.", ex);
		}

		// for every subsegment, add a LocalID to the XRD

		for (SubSegment localSubSegment : localSubSegments) {

			String localSubSegmentName = localSubSegment.getName();

			if (this.excludeSelf && localSubSegmentName.equals(subSegmentName)) continue;

			xrd.addLocalID(new LocalID(localSubSegmentName));
		}

		// done

		return(xrd);
	}
}
