package org.openxri.proxy.impl;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.TreeSet;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openxri.XRI;
import org.openxri.XRIParseException;
import org.openxri.config.ProxyConfig;
import org.openxri.proxy.Proxy;
import org.openxri.proxy.ProxyException;
import org.openxri.resolve.MimeType;
import org.openxri.resolve.Resolver;
import org.openxri.resolve.ResolverCache;
import org.openxri.resolve.ResolverFlags;
import org.openxri.resolve.ResolverState;
import org.openxri.resolve.TrustType;
import org.openxri.resolve.exception.IllegalTrustTypeException;
import org.openxri.resolve.exception.PartialResolutionException;
import org.openxri.util.URLUtils;
import org.openxri.xml.Status;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;


/**
 * Provides a servlet implementation for the XRI resolution protocol
 *
 * @author =wil
 * @author srinivasa.adapa@neustar.com
 */
public abstract class AbstractProxy implements Proxy
{

	protected ProxyConfig config;

	/**
	 * Static Logging object that can be used by derived classes
	 */
	protected static org.apache.commons.logging.Log log =
		org.apache.commons.logging.LogFactory.getLog(
				AbstractProxy.class.getName());

	/* static constants for proxy */
	public static final String _XRD_R = "_xrd_r";
	public static final String _XRD_T = "_xrd_t";
	public static final String _XRD_M = "_xrd_m";
	public static final String HTTP_ERROR_CONTENT_TYPE = "text/plain; charset=UTF-8";
	public static final String HTTP_XML_CONTENT_TYPE = "text/xml; charset=UTF-8";

	protected boolean supportXRDS     = false;
	protected boolean supportXRD      = false;
	protected boolean supportURIList  = false;
	protected boolean supportRedirect = false;

	protected String rootRedirect = null;
	protected String bareXRINotFoundRedirect = null;
	/**
	 * The XRI resolver object for the server.  Used for proxied resolution.
	 */
	protected Resolver resolver = null;


	public AbstractProxy(ProxyConfig config) {
		this.config = config;
	}


	public void init() throws ProxyException {

		this.resolver = new Resolver();

		this.resolver.setMaxFollowRedirects(this.config.getMaxFollowRedirects());
		this.resolver.setMaxFollowRefs(this.config.getMaxFollowRefs());
		this.resolver.setMaxRequests(this.config.getMaxRequests());
		this.resolver.setMaxTotalBytes(this.config.getMaxTotalBytes());
		this.resolver.setMaxBytesPerRequest(this.config.getMaxBytesPerRequest());

		try {
			this.resolver.setAuthority("=", this.config.getEqualsAuthority());
			this.resolver.setAuthority("@", this.config.getAtAuthority());
			this.resolver.setAuthority("!", this.config.getBangAuthority());
		} catch (Exception ex) {
			throw new ProxyException("Cannot initialize Resolver. Check the =, @ and ! root authorities.", ex);
		}

		String[] supports = this.config.getSupportedResMediaTypes();
		if (supports != null) {
			for (int i = 0; i < supports.length; i++) {
				String type = supports[i].trim().toLowerCase();
				if (type.equals(MimeType.URI_LIST)) {
					this.supportURIList = true;
				}
				else if (type.equals(MimeType.XRD_XML)) {
					this.supportXRD = true;
				}
				else if (type.equals(MimeType.XRDS_XML)) {
					this.supportXRDS = true;
				}
				else if (type.equals("redirect")) {
					this.supportRedirect = true;
				}
				else {
					log.warn("unknown resolution media type: " + type);
				}
			}
		}

		String[] httpsBypassAuthorities = this.config.getHttpsBypassAuthorities();
		for (int i = 0; i < httpsBypassAuthorities.length; i++) {
			XRI x = XRI.fromURINormalForm(httpsBypassAuthorities[i]);
			this.resolver.addHttpsBypassAuthority(httpsBypassAuthorities[i]);
		}

		String[] samlBypassAuthorities = this.config.getSamlBypassAuthorities();
		for (int i = 0; i < samlBypassAuthorities.length; i++) {
			XRI x = XRI.fromURINormalForm(samlBypassAuthorities[i]);
			this.resolver.addSAMLBypassAuthority(samlBypassAuthorities[i]);
		}

		if (this.config.getUseCache()) {
			this.resolver.setCache(new ResolverCache());
			this.resolver.setDefaultCacheTTL(this.config.getDefaultCacheTTL());
			this.resolver.setMinCacheTTL(this.config.getMinCacheTTL());
			this.resolver.setMaxCacheTTL(this.config.getMaxCacheTTL());
			this.resolver.setNegativeCacheTTL(this.config.getNegativeCacheTTL());
		}
		
		this.rootRedirect = this.config.getRootRedirect();
		this.bareXRINotFoundRedirect = this.config.getBareXRINotFoundRedirect();
	}

	public void shutdown() {

	}

	/**
	 * Service an incoming request.
	 */
	public void process(HttpServletRequest request, HttpServletResponse response)
	{
		log.trace("process - enter");

		try {
			if(this.resolver == null) {

				log.error("proxy not initialized check configuration");
				sendError(request, response, null, HttpServletResponse.SC_INTERNAL_SERVER_ERROR, 300, "TEMPORARY_FAIL(code=300): proxy not available", null);
				return;
			}

			// get the QXRI to resolve from the REQUEST_URI - ContextPath
			String requestURI = request.getRequestURI();
			String ctxPath = request.getContextPath();
			String servletPath = request.getServletPath();

			// security checks
			if (!ctxPath.equalsIgnoreCase(requestURI.substring(0, ctxPath.length()))) {
				sendError(request, response, null, HttpServletResponse.SC_NOT_FOUND, 210, "INVALID_INPUT(code=210), invalid HXRI usage", null);
				return;
			}
			if (!servletPath.equalsIgnoreCase(requestURI.substring(ctxPath.length(), servletPath.length() + ctxPath.length()))) {
				sendError(request, response, null, HttpServletResponse.SC_NOT_FOUND, 210, "INVALID_INPUT(code=210), invalid HXRI usage", null);
				return;
			}
			String sPath = requestURI.substring(ctxPath.length() + servletPath.length());


			log.debug("requestURL=" + request.getRequestURL());
			log.debug("requestURI=" + request.getRequestURI());
			log.debug("contextPath=" + request.getContextPath());
			log.debug("pathinfo=" + request.getPathInfo());
			log.debug("servletPath=" + request.getServletPath());
			log.debug("pathTranslated=" + request.getPathTranslated());


			// Must have a path, with more than a starting slash
			if ((sPath != null) && (sPath.length() > 1) && (sPath.charAt(0) == '/')) {
				sPath = sPath.substring(1); // Trim the slash
			}
			else {
				if (this.rootRedirect!= null) {
					response.sendRedirect(this.rootRedirect);
					return;
				}
				// Send not found and bail
				sendError(request, response, null, HttpServletResponse.SC_NOT_FOUND, 210, "INVALID_INPUT(code=210), no path exists", null);
				return;
			}

			log.trace("Checking path (" + sPath + ") to see if auto-correction is necessary");
			String autoCorrected = checkAutoCorrect(sPath);
			if (autoCorrected != null) {
				String newURL = buildAbsoluteURL(request, autoCorrected);
				log.debug("Redirecting to URL " + newURL);
				response.sendRedirect(newURL);
				return;
			}
			log.trace("Auto-correction not needed");

			// initialize the following input parameters to proxy server in addition to 
			// addition flags refs, trust & sep
			MimeType resolutionMediaType = null;
			String serviceResMediaType = null;
			String serviceType = null;  // this always comes from HTTP get query

			serviceResMediaType = getServiceMediaTypeFromAcceptHeader(request);
			QueryParams qp = parseQuery(request);

			if (qp.xrdR != null && qp.xrdR.trim().length() > 0) {
				MimeType mType = MimeType.parse(qp.xrdR);
				if (mType != null && mType.getType() != null && mType.getType().length() > 0) {
					resolutionMediaType = mType;
				}
			}

			if (qp.xrdT != null && (qp.xrdT.trim().length() == 0))
				serviceType = null;
			else
				serviceType = qp.xrdT;

			if (qp.xrdM != null && (qp.xrdM.trim().length() > 0)) {
				// this overrides earlier value (that might be in Accept: header)
				serviceResMediaType = qp.xrdM;
			}

			// reconstruct the QXRI (without _xrd_* resolution query parameters)
			if (qp.opaque != null) {
				sPath = sPath + '?' + qp.opaque;
			}

			processProxyRequest(sPath, resolutionMediaType, serviceType, serviceResMediaType, request, response);
		}
		catch (Throwable the)
		{
			// log and send a 500
			log.fatal("Received RuntimeExeption while processing request: path="+request.getPathInfo() + " error: "+the);
			the.printStackTrace();
			sendFatalError(response, the.getMessage());
		}
	}

	private String getServiceMediaTypeFromAcceptHeader(HttpServletRequest request) {
		Enumeration acceptValues = request.getHeaders(Tags.HEADER_ACCEPT);
		TreeSet ss = new TreeSet();

		while (acceptValues.hasMoreElements()) {
			String headerValue = (String)acceptValues.nextElement();
			String[] v = headerValue.split("\\s*,\\s*");
			for (int i = 0; i < v.length; i++) {
				log.trace("getServiceMediaTypeFromHeader: checking header '" + v[i] + "'");
				MimeType m = MimeType.parse(v[i]);
				ss.add(m);
			}
		}

		if (ss.size() > 0) {
			MimeType m = (MimeType)ss.first();
			return m.getType();
		}
		return null;
	}

	private boolean checkSupportedMediaTypes(HttpServletRequest oReq, HttpServletResponse oResp, MimeType resMediaType)
	throws IOException
	{
		String notImplemented = "NOT_IMPLEMENTED(code=201): Resolution media type '" + resMediaType + "' is not supported by this proxy";

		if (resMediaType == null) {
			if (! this.supportRedirect) {
				sendError(oReq, oResp, null, 201, "NOT_IMPLEMENTED(code=201): HTTP redirect resolution mechanism is not supported by this proxy", null);
				return false;
			}
			return true;
		}

		if (!resMediaType.isValidXriResMediaType()) {
			sendError(oReq, oResp, null, 212, "INVALID_RESOLUTION_MEDIA_TYPE(code=212): Unknown media type '" + resMediaType + "'", null);
			return false;
		}

		// must be one of the types
		if (resMediaType.isType(MimeType.XRDS_XML) && !this.supportXRDS) {
			sendError(oReq, oResp, null, 201, notImplemented, null);
			return false;
		}
		else if (resMediaType.isType(MimeType.XRD_XML) && !this.supportXRD) {
			sendError(oReq, oResp, null, 201, notImplemented, null);
			return false;
		}
		else if (resMediaType.isType(MimeType.URI_LIST) && !this.supportURIList) {
			sendError(oReq, oResp, null, 201, notImplemented, null);
			return false;
		}
		return true;
	}

	/**
	 * Process a request for proxied resolution
	 */
	protected void processProxyRequest(
			String qxri,
			MimeType resMediaType,
			String serviceType,
			String serviceMediaType,
			HttpServletRequest request,
			HttpServletResponse response)
	throws IOException
	{
		log.trace("processProxyRequest - enter");
		
		// check if we support the media types
		if (!checkSupportedMediaTypes(request, response, resMediaType)) {
			log.trace("processProxyRequest - checkSupportedMediaTypes returned false, returning.");
			return;
		}

		// Toss if something is amiss
		if ((qxri == null) || (qxri.length() == 0)) {
			log.debug("processProxyRequest - sXRI is null or empty");
			sendError(request, response, null, HttpServletResponse.SC_INTERNAL_SERVER_ERROR, 211, "INVALID_QXRI(code=211): null or empty", null);
			return;
		}

		XRI oXRI = null;
		try {
			oXRI = XRI.fromURINormalForm(qxri);
		}
		catch (XRIParseException oEx) {
			// log and send a 404
			log.warn("Error constructing XRI: " + qxri + ", " + oEx);
			sendError(request, response, qxri, 211, "INVALID_QXRI(code=211): "+oEx.getMessage(), oEx);
			return;
		}

		// defaults if resolution media type is null
		TrustType trustType = new TrustType();
		boolean refs = true;
		boolean sep = true;
		boolean https = false;
		boolean saml = false;
		boolean cid = true;
		boolean uric = false;
		boolean nodefault_t = false;
		boolean nodefault_p = false;
		boolean nodefault_m = false;
		boolean debug = false;

		String tempStr = null;
		if(resMediaType != null) {
			tempStr = resMediaType.getParam(MimeType.PARAM_REFS);
			if(tempStr != null && tempStr.equals("false"))
				refs = false;

			tempStr = resMediaType.getParam(MimeType.PARAM_SEP);
			if(tempStr != null && tempStr.equals("false"))
				sep = false;

			tempStr = resMediaType.getParam(MimeType.PARAM_TRUST);
			if (tempStr != null) {
				try {
					trustType.setType(tempStr);
				}
				catch (IllegalTrustTypeException e) {
					sendError(request, response, qxri, 212, "INVALID_RESOLUTION_MEDAIA_TYPE(code=212): " + resMediaType, e);
					return;
				}

				https = trustType.isHTTPS();
				saml = trustType.isSAML();
			}

			tempStr = resMediaType.getParam(MimeType.PARAM_HTTPS);
			if(tempStr != null && tempStr.equals("true"))
				https = true;

			tempStr = resMediaType.getParam(MimeType.PARAM_SAML);
			if(tempStr != null && tempStr.equals("true"))
				saml = true;

			// override the trusttype
			trustType.setParameterPair(https, saml);

			tempStr = resMediaType.getParam(MimeType.PARAM_URIC);
			if(tempStr != null && tempStr.equals("true"))
				uric = true;

			tempStr = resMediaType.getParam(MimeType.PARAM_CID);
			if(tempStr != null && tempStr.equals("false"))
				cid = false;

			tempStr = resMediaType.getParam(MimeType.PARAM_NO_DEFAULT_M);
			if(tempStr != null && tempStr.equals("true"))
				nodefault_m = true;

			tempStr = resMediaType.getParam(MimeType.PARAM_NO_DEFAULT_P);
			if(tempStr != null && tempStr.equals("true"))
				nodefault_p = true;

			tempStr = resMediaType.getParam(MimeType.PARAM_NO_DEFAULT_T);
			if(tempStr != null && tempStr.equals("true"))
				nodefault_t = true;

			tempStr = resMediaType.getParam("debug");
			if(tempStr != null && (tempStr.equals("true") || tempStr.equals("1")))
				debug = true;

		}
		
		// give subclasses a chance to do something
		if (onBeforeResolution(qxri, request, response)) {
			log.trace("processProxyRequest - onBeforeResolution returned true, assuming request is fully handled, returning");
			return;
		}

		// set the request preferences on the resolver before querying.
		ResolverState state = new ResolverState();
		ResolverFlags flags = new ResolverFlags();
		flags.setCid(cid);
		flags.setHttps(https);
		flags.setSaml(saml);
		flags.setRefs(refs);
		flags.setUric(uric);
		flags.setNoDefaultM(nodefault_m);
		flags.setNoDefaultP(nodefault_p);
		flags.setNoDefaultT(nodefault_t);

		// resolve
		XRDS xrds = null;
		XRD xrd = null;
		try
		{
			if (sep) {
				if (resMediaType == null) {
					// see section 7.6 for special redirect rule
					//					ArrayList<?> uris = this.resolver.resolveSEPToURIList(oXRI.toString(), trustType, serviceType, serviceMediaType, refs);
					ArrayList<?> uris = this.resolver.resolveSEPToURIList(oXRI, serviceType, serviceMediaType, flags, state);
					if (uris == null || uris.size() == 0) {
						sendError(request, response, qxri, 241, "SEP_NOT_FOUND(code=241): no url found", null);
						return;
					}
					String s = (String) uris.get(0);
					if (onResolutionSuccess(qxri, state, s, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					log.trace("Sending redirect to '" + s + "'");
					if (this.config.getSetting("RedirectCode").equals("301"))
						send301(response, s);
					else if (this.config.getSetting("RedirectCode").equals("303"))
						send303(response, s);
					else
						response.sendRedirect(s);
				}
				else if (resMediaType.isType(MimeType.URI_LIST)) {
					//					String  text = this.resolver.resolveSEPToTextURIList(oXRI.toString(), trustType, serviceType, serviceMediaType, refs);
					String text = this.resolver.resolveSEPToTextURIList(oXRI, serviceType, serviceMediaType, flags, state);
					if (onResolutionSuccess(qxri, state, text, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					if (text.length() <= 0)
						sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "SEP_NOT_FOUND(code=241): no url found", null);
					else
						sendResponse(response, resMediaType.getType(), text, null);
				}
				else if (resMediaType.isType(MimeType.XRDS_XML)) {
					//					xrds = this.resolver.resolveSEPToXRDS(oXRI, trustType, serviceType, serviceMediaType, refs);
					xrds = this.resolver.resolveSEPToXRDS(oXRI, serviceType, serviceMediaType, flags, state);
					if (onResolutionSuccess(qxri, state, xrds, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					sendResponse(response, debug, resMediaType.getType(), xrds.toString(), trustType);
				}
				else if (resMediaType.isType(MimeType.XRD_XML)) {
					//					xrd = this.resolver.resolveSEPToXRD(oXRI, trustType, serviceType, serviceMediaType, refs);
					xrd = this.resolver.resolveSEPToXRD(oXRI, serviceType, serviceMediaType, flags, state);
					if (onResolutionSuccess(qxri, state, xrd, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					sendResponse(response, debug, resMediaType.getType(), xrd.toResultString(), trustType);
				}
				else {
					// else - we should have taken care of it in checkSupportedMediaTypes
					log.error("processProxyRequest - should not reach here (sep=true)");
				}
			}
			else {
				//// authority resolution only
				if(resMediaType == null) {
					resMediaType = new MimeType(MimeType.XRDS_XML);
				}

				if (resMediaType.isType(MimeType.XRDS_XML)) {
					//					xrds = this.resolver.resolveAuthToXRDS(oXRI, trustType, refs);
					xrds = this.resolver.resolveAuthToXRDS(oXRI, flags, state);
					if (onResolutionSuccess(qxri, state, xrds, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					sendResponse(response, debug, resMediaType.getType(), xrds.toString(), trustType);
					return;
				}
				else if (resMediaType.isType(MimeType.XRD_XML)) {
					//					xrd = this.resolver.resolveAuthToXRD(oXRI, trustType, refs);
					xrd = this.resolver.resolveAuthToXRD(oXRI, flags, state);
					if (onResolutionSuccess(qxri, state, xrd, request, response)) {
						log.trace("processProxyRequest - onResolutionSuccess returned true, assuming request is fully handled, returning");
						return;
					}
					sendResponse(response, debug, resMediaType.getType(), xrd.toString(), trustType);
				}
				else if (resMediaType.isType(MimeType.URI_LIST)) {
					// ignore (must do SEP when text/uri-list is specified)
					log.warn("text/uri-list given but does not want to do service selection");
					// TODO: do we return an error?
				}
				else {
					// else - we should have taken care of it in checkSupportedMediaTypes
					log.error("processProxyRequest - should not reach here (sep=false)");
				}
			}
			return;
		}
		catch (PartialResolutionException pre) {
			log.info("processProxyRequest - caught PartialResolutionException");
			sendPartialResponse(request, response, qxri, debug, resMediaType, pre, trustType);
		}
	}

	/**
	 * Checks to see if auto-correction is necessary. If so, return the auto-corrected string.
	 * @param s
	 * @return <code>null</code> if no auto-correction needs to be done, or the auto-corrected string otherwise.
	 */
	private static String checkAutoCorrect(String s)
	{
		int i = 0;
		if (s.length() > 6 && s.substring(0, 6).equalsIgnoreCase("xri://"))
			i = 6;
		i = s.indexOf('/', i);
		if (i == -1)
			return null; // no path, ok

		String path = s.substring(i+1);
		String pathQuoted = quotePath(path);
		if (pathQuoted.equals(path))
			return null; // no need to quote

		// combine quoted path
		String newXRI = s.substring(0, i+1) + pathQuoted;
		return newXRI;
	}

	/**
	 * auto correction of input path parameter
	 */
	private static String quotePath (String path) {
		if (path == null || path.length() == 0)
			return path;

		char first = path.charAt(0);
		switch (first)
		{
		case '+':
		case '=':
		case '@':
		case '$':
		case '!':
		{
			// starts with GCS character, we need to parenthesize
			int i = path.indexOf('/');
			if (i == -1)
				// quote entire path since there is only one path segment
				return "(" + path + ")";
			else
				// insert the closing parenthesis right before the '/'
				return "(" + path.substring(0, i) + ")" + path.substring(i);
		}
		}
		return path;
	}

	private static String buildAbsoluteURL(HttpServletRequest request, String relPath)
	{
		StringBuffer sb = new StringBuffer();
		sb.append(request.getContextPath());

		log.trace("context='" + request.getContextPath() + "'");
		log.trace("path='" + request.getServletPath() + "'");

		log.trace("sb='" + sb + "'");

		int len = sb.length();
		if (len > 1 && sb.charAt(len-1) != '/')
			sb.append('/');

		log.trace("sb2='" + sb + "'");

		sb.append(request.getServletPath());
		len = sb.length();
		if (len > 1 && sb.charAt(len-1) != '/')
			sb.append('/');

		if (sb.length() == 0)
			sb.append('/');

		log.trace("sb3='" + sb + "'");

		sb.append(relPath);
		if (request.getQueryString() != null) {
			sb.append('?');
			sb.append(request.getQueryString());
		}
		log.trace("sb4='" + sb + "'");
		return sb.toString();
	}

	/**
	 * This method is called when an unknown error is encountered.
	 * This simply outputs the error message in text/plain.
	 */
	protected void sendFatalError(HttpServletResponse resp, String errorMessage){
		resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
		resp.setContentType(HTTP_ERROR_CONTENT_TYPE);
		try {
			ServletOutputStream out = resp.getOutputStream();
			out.println("The proxy has encountered a fatal error. Internal error message follows: ");
			out.println(errorMessage);
		}
		catch (IOException e) {
		}
	}

	protected void sendError(
		HttpServletRequest request,
		HttpServletResponse response,
		String qxri,
		int errorCode, 
		String errorMessage,
		Throwable the) throws IOException {
	
		sendError(request, response, qxri, HttpServletResponse.SC_OK, errorCode, errorMessage, the);
	}
	
	protected void sendError(
		HttpServletRequest request,
		HttpServletResponse response, 
		String qxri,
		int statusCode,
		int errorCode,
		String errorMessage,
		Throwable the) throws IOException {
		
		sendResponse(response, HTTP_ERROR_CONTENT_TYPE, errorMessage, null);
	}
	
	protected void sendPartialResponse(
			HttpServletRequest request, 
			HttpServletResponse response, 
			String qxri, 
			boolean isDebug, 
			MimeType resMediaType, 
			PartialResolutionException pre,
			TrustType trustType)
	throws IOException
	{
		log.trace("sendPartialResponse(debug=" + isDebug + ", partialXRDS=" + pre.getPartialXRDS().toString());

		XRDS partialXRDS = pre.getPartialXRDS();
		if (this.bareXRINotFoundRedirect != null) {
			if (resMediaType == null & partialXRDS.getFinalXRD().getStatusCode().equals(Status.QUERY_NOT_FOUND)) {
				response.sendRedirect(this.bareXRINotFoundRedirect + URLEncoder.encode(qxri, "UTF-8"));
				return;
			}
		}
		response.setStatus(HttpServletResponse.SC_OK);
		XRD xrd = partialXRDS.getFinalXRD();
		String errMsg = "";
		if (xrd != null) {
			Status stat = xrd.getStatus();
			errMsg = "Error code: " + stat.getCode() + " - " + stat.getText();
		}

		if (resMediaType == null) {
			sendError(request, response, qxri, Integer.parseInt(xrd.getStatus().getCode()), errMsg, pre);
		}
		else {
			if (this.onPartialResolutionSuccess(qxri, request, response, Integer.parseInt(xrd.getStatus().getCode()), errMsg, pre)) {
				log.trace("sendPartialResponse - onPartialResolutionSuccess returned true, assuming request is fully handled, returning");
				return;
			}
			if (resMediaType.isType(MimeType.URI_LIST)) {
				sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "# " + errMsg, trustType);
			}
			else if (resMediaType.isType(MimeType.XRDS_XML)) {
				sendResponse(response, isDebug, resMediaType.getType(), partialXRDS.toString(), trustType);
			}
			else if (resMediaType.isType(MimeType.XRD_XML)) {
				sendResponse(response, isDebug, resMediaType.getType(), xrd.toString(), trustType);
			}
			else {
				// else - we should have taken care of it in checkSupportedMediaTypes
				log.error("processProxyRequest - should not reach here (exception)");
			}
		}
	}

	/**
	 * Send a 301 HTTP redirect to the client.
	 */
	protected void send301(
			HttpServletResponse response,
			String location)
	throws IOException
	{
		response.setStatus(HttpServletResponse.SC_MOVED_PERMANENTLY);
		response.addHeader("Location", location);
	}

	/**
	 * Send a 303 HTTP redirect to the client.
	 */
	protected void send303(
			HttpServletResponse response,
			String location)
	throws IOException
	{
		response.setStatus(HttpServletResponse.SC_SEE_OTHER);
		response.addHeader("Location", location);
	}

	/**
	 * Send a HTTP response to the client.
	 * This method is designed to be overridden by subclasses for user-friendly
	 * display of errors.
	 * 
	 * @param isDebug flag to indicate if the client wants text/xml to be returned
	 */
	protected void sendResponse(
			HttpServletResponse response,
			boolean isDebug,
			String contentType,
			String result,
			TrustType trustType)
	throws IOException
	{
		if (isDebug)
			sendResponse(response, HTTP_XML_CONTENT_TYPE, result, trustType);
		else
			sendResponse(response, contentType, result, trustType);
	}

	/**
	 * Send a HTTP response to the client.
	 */
	protected void sendResponse(
			HttpServletResponse response, 
			String contentType, 
			String result, 
			TrustType trustType)
	throws IOException
	{
		sendResponse(response, HttpServletResponse.SC_OK, contentType, result, trustType);
	}

	/**
	 * Send a HTTP response with the specified status code to the client.
	 */
	protected void sendResponse(
			HttpServletResponse response, 
			int statusCode,
			String contentType, 
			String result, 
			TrustType trustType)
	throws IOException
	{
		response.setStatus(statusCode);
		if (trustType != null)
			contentType += ";" + trustType.getParameterPair();
		response.setContentType(contentType);
		response.getOutputStream().write(result.getBytes("UTF-8"));
	}

	protected QueryParams parseQuery(HttpServletRequest req)
	{
		QueryParams qp = new QueryParams();
		String queryString = req.getQueryString();
		String xrdR = null;
		String xrdT = null;
		String xrdM = null;
		StringBuffer opaque = new StringBuffer();

		if (queryString == null) {
			return qp;
		}

		log.trace("parseQuery() - queryString.length=" + queryString.length());

		int start = 0;
		while (start < queryString.length()) {
			log.trace("parseQuery() - start=" + start);
			String kvpair;
			int i = queryString.indexOf('&', start);
			log.trace("parseQuery() - i=" + i);
			if (i == -1) {
				kvpair = queryString.substring(start);
				start = queryString.length(); // done
			}
			else {
				kvpair = queryString.substring(start, i);
				start = i+1;
			}

			// parse kvpair
			int eq = kvpair.indexOf('=');
			String key, val;
			if (eq == -1) {
				key = kvpair;
				val = "";
			}
			else {
				key = kvpair.substring(0, eq);
				val = kvpair.substring(eq+1);
			}

			// see if we recognize the key
			if (xrdR == null && key.toLowerCase().equals(_XRD_R)) {
				log.trace("parseQuery() - xrdR=" + val);
				xrdR = val;
			}
			else if (xrdT == null && key.toLowerCase().equals(_XRD_T)) {
				log.trace("parseQuery() - xrdT=" + val);
				xrdT = val;
			}
			else if (xrdM == null && key.toLowerCase().equals(_XRD_M)) {
				log.trace("parseQuery() - xrdM=" + val);
				xrdM = val;
			}
			else {
				log.trace("parseQuery() - param=" + kvpair);
				// don't recognize it or we already have the parameter
				// just add it to the opaque string
				if (opaque.length() > 0)
					opaque.append('&');

				opaque.append(kvpair);
			}
		}

		qp.xrdR = URLUtils.decode(xrdR);
		qp.xrdT = URLUtils.decode(xrdT);
		qp.xrdM = URLUtils.decode(xrdM);
		qp.opaque = (opaque.length() > 0)? opaque.toString(): null;

		log.trace("parseQuery() - xrdR=" + qp.xrdR + ", xrdT=" + qp.xrdT + ", xrdM="+ qp.xrdM + ", opaque="+qp.opaque);
		return qp;
	}
	
	/*
	 * abstract methods to be implemented by subclasses
	 */

	/**
	 * Hook method which gets called BEFORE resolution takes place
	 * @param qxri The XRI to be resolved
	 * @param request The http servlet request
	 * @param response The http servlet response
	 * @return True means the entire request has been completely handled by this method and
	 * nothing else will be done
	 */
	public abstract boolean onBeforeResolution(String qxri, HttpServletRequest request, HttpServletResponse response);

	/**
	 * Hook method which gets called AFTER resolution has been successful
	 * @param qxri The XRI to be resolved
	 * @param state The resolution state after resolution
	 * @param result The resolution result (String or XRDS or XRD)
	 * @param request The http servlet request
	 * @param response The http servlet response
	 * @return True means the entire request has been completely handled by this method and
	 * nothing else will be done
	 */
	public abstract boolean onResolutionSuccess(String qxri, ResolverState state, Object result, HttpServletRequest request, HttpServletResponse response);

	/**
	 * Hook method which gets called AFTER resolution has only been partially successful
	 * @param qxri The XRI to be resolved
	 * @param request The http servlet request
	 * @param response The http servlet response
	 * @return True means the entire request has been completely handled by this method and
	 * nothing else will be done
	 */
	public abstract boolean onPartialResolutionSuccess(String qxri, HttpServletRequest request, HttpServletResponse response, int errorCode, String errorMessage, PartialResolutionException pre);
}
