package org.openxri.proxy.impl;


import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.StringTokenizer;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openxri.XRI;
import org.openxri.XRIParseException;
import org.openxri.config.ProxyConfig;
import org.openxri.proxy.Proxy;
import org.openxri.proxy.ProxyException;
import org.openxri.resolve.MimeType;
import org.openxri.resolve.Resolver;
import org.openxri.resolve.TrustType;
import org.openxri.resolve.exception.IllegalTrustTypeException;
import org.openxri.resolve.exception.PartialResolutionException;
import org.openxri.util.URLUtils;
import org.openxri.xml.AuthenticationService;
import org.openxri.xml.Service;
import org.openxri.xml.Status;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;


/**
 * Provides a servlet implementation for the XRI resolution protocol
 *
 * @author =wil
 * @author srinivasa.adapa@neustar.com
 * 
 * 
 * In the follow-up of the May 2007 OASIS F2F Meeting in San Diego, it was discussed whether an OpenID
 * relying party could be tricked into accepting any HXRI as an OpenID. The idea was to have the proxy
 * check if the client sends a User-Agent header. If that header was missing, the proxy would assume the
 * request comes from an OpenID relying party instead of a webbrowser, and it would output a
 * Yadis-compliant HTML document that every OpenID RP understands. The idea was later dropped
 * because this method was found to be too unreliable.
 *
 * @author =peacekeeper
 */
public class OpenIDHackProxy implements Proxy
{

	protected ProxyConfig config;

	/**
	 * Static Logging object that can be used by derived classes
	 */
	protected static org.apache.commons.logging.Log log =
		org.apache.commons.logging.LogFactory.getLog(
				OpenIDHackProxy.class.getName());

	/* static constants for proxy */
	public static final String _XRD_R = "_xrd_r";
	public static final String _XRD_T = "_xrd_t";
	public static final String _XRD_M = "_xrd_m";
	public static final String HTTP_ERROR_CONTENT_TYPE = "text/plain; charset=UTF-8";
	public static final String HTTP_XML_CONTENT_TYPE = "text/xml; charset=UTF-8";

	protected boolean supportXRDS     = false;
	protected boolean supportXRD      = false;
	protected boolean supportURIList  = false;
	protected boolean supportRedirect = false;

	/**
	 * The XRI resolver object for the server.  Used for proxied resolution.
	 */
	protected Resolver resolver = null;

	public OpenIDHackProxy(ProxyConfig config) {

		this.config = config;
	}

	public void init() throws ProxyException
	{
		resolver = new Resolver();

		resolver.setMaxFollowRedirects(config.getMaxFollowRedirects());
		resolver.setMaxFollowRefs(config.getMaxFollowRefs());
		resolver.setMaxRequests(config.getMaxRequests());
		resolver.setMaxTotalBytes(config.getMaxTotalBytes());
		resolver.setMaxBytesPerRequest(config.getMaxBytesPerRequest());

		try {

			resolver.setAuthority("=", config.getEqualsAuthority());
			resolver.setAuthority("@", config.getAtAuthority());
			resolver.setAuthority("!", config.getBangAuthority());
		} catch (Exception ex) {

			throw new ProxyException("Cannot initialize Resolver. Check the =, @ and ! root authorities.", ex);
		}

		String[] supports = config.getSupportedResMediaTypes();
		if (supports != null) {

			for (int i = 0; i < supports.length; i++) {

				String type = supports[i].trim().toLowerCase();
				if (type.equals(MimeType.URI_LIST)) {
					supportURIList = true;
				}
				else if (type.equals(MimeType.XRD_XML)) {
					supportXRD = true;
				}
				else if (type.equals(MimeType.XRDS_XML)) {
					supportXRDS = true;
				}
				else if (type.equals("redirect")) {
					supportRedirect = true;
				}
				else {
					log.warn("unknown resolution media type: " + type);
				}
			}
		}
	}

	public void shutdown() {
		
	}

	/**
	 * Service an incoming request.
	 */
	public void process(HttpServletRequest request, HttpServletResponse response)
	{
		boolean isDumbBrowser = true;
		log.trace("doGet() - enter");
		log.debug("Request: " + request.getRequestURI());

		try {
			if(this.resolver == null) {

				log.error("proxy not initialized check configuration");
				sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "TEMPORARY_FAIL(code=300): proxy not available", null);
				return;
			}

			// get the QXRI to resolve from the REQUEST_URI - ContextPath
			String context = request.getContextPath();
			String sPath = request.getRequestURI().substring(context.length());

			// Must have a path, with more than a starting slash
			if ((sPath != null) && (sPath.length() > 1) && (sPath.charAt(0) == '/')) {
				sPath = sPath.substring(1); // Trim the slash
			}
			else {
				// Send not found and bail
				sendResponse(response, HttpServletResponse.SC_NOT_FOUND, HTTP_ERROR_CONTENT_TYPE, "INVALID_INPUT(code=210), no path exists", null);
				return;
			}

			log.trace("Checking path (" + sPath + ") to see if auto-correction is necessary");
			String autoCorrected = checkAutoCorrect(sPath);
			if (autoCorrected != null) {
				String newURL = buildAbsoluteURL(request, autoCorrected);
				log.debug("Redirecting to URL " + newURL);
				response.sendRedirect(newURL);
				return;
			}
			log.trace("Auto-correction not needed");


			String sAccept = request.getHeader(Tags.HEADER_ACCEPT);
			// initialize the following input parameters to proxy server in addition to 
			// addition flags refs, trust & sep
			MimeType resolutionMediaType = null;
			String serviceResMediaType = null;
			String serviceType = null;  // this always comes from http get query

			/*
			 * We expect Accept header of format and would contain only expected variables and values
			 * The general format of aceept header is: <mediatype>[;name=value]*[,<mediatype>[;named=value]*]
			 * 1. If multiple mediatypes appears the proxy server shoud pick the first one.
			 * 2. If mediatype is one of the application/xrd(s)+xml or text/uri-list then ResolutionMediaType=mediatype
			 * and ServiceMediaType becomes null
			 * 3. If mediatype doesn't match with applciation/xrd(s) or text/uri-list ResolutionMediaType=null
			 * and ServiceMediaType = mediatype
			 * if accept header is null then ResolutionMediaType and ServiceMediaType = null (used loose constraint)
			 * 4. In any case the proxy query parameters _xrd_r and _xrd_m takes higher precendence over accept header
			 */
			if (sAccept != null ){
				StringTokenizer tokenizer = new StringTokenizer(sAccept, ",");
				while (tokenizer.hasMoreTokens() ){
					String token = (String)tokenizer.nextToken();

					MimeType mType = MimeType.parse(token);
					if (mType == null || mType.getType() == null || mType.getType().length() == 0)
						continue; // ignore

					if (mType.isValidXriResMediaType()) {
						resolutionMediaType = mType;
						serviceResMediaType = null;
						isDumbBrowser = false;
					}
					else {
						resolutionMediaType = null;
						serviceResMediaType = mType.toString(); 
					}
					break;
				}
			}

			QueryParams qp = parseQuery(request);


			if (qp.xrdR != null) {
				if (qp.xrdR.trim().length() == 0) {
					// parameter exists, but is empty
					resolutionMediaType = null;
				}
				else {
					MimeType mType = MimeType.parse(qp.xrdR);
					if (mType != null && mType.getType() != null && mType.getType().length() > 0) {
						// this overrides earlier value (that might be in Accept: header)
						resolutionMediaType = mType;
					}
				}
			}


			if (qp.xrdT != null && (qp.xrdT.trim().length() == 0))
				serviceType = null;
			else
				serviceType = qp.xrdT;


			if (qp.xrdM != null && (qp.xrdM.trim().length() == 0)){
				serviceResMediaType = qp.xrdM;
			}

			// reconstruct the QXRI (without _xrd_* resolution query parameters)
			if (qp.opaque != null) {
				sPath = sPath + '?' + qp.opaque;
			}

			// Here comes the hack. If there is no User-Agent header, we think the request comes from
			// an OpenID server trying to do discovery, instead of a webbrowser. In this case, we output a hacky
			// HTML document with OpenID delegation in its <head>.
			
			log.debug("User-Agent: " + request.getHeader("User-Agent"));
			
			if (request.getHeader("User-Agent") == null) {
				
				String server;
				String identifier = request.getRequestURI().substring(request.getContextPath().length() + 1);
					
				XRD xrd = resolver.resolveSEPToXRD(
						sPath, 
						new TrustType(TrustType.TRUST_NONE), 
						AuthenticationService.SERVICE_TYPE1, 
						null, 
						true);

				if (xrd == null || xrd.getSelectedServices().getList().size() == 0) {

					sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "SEP_NOT_FOUND(code=241): no authentication url found", null);
					return;
				}

				server = ((Service) xrd.getSelectedServices().getList().get(0)).getURIAt(0).getUriString();

				log.debug("OpenID hack triggered: Server=" + server + ", Identifier=" + identifier);
				
				this.outputOpenIDServerHack(response, server);
				return;
			}
			
			// process the request
			
			processProxyRequest(sPath, resolutionMediaType, serviceType, serviceResMediaType, request, response, isDumbBrowser);
		}
		catch (Throwable the)
		{
			// log and send a 500
			log.fatal("Received RuntimeExeption while processing request: path="+request.getPathInfo() + " error: "+the);
			the.printStackTrace();
			sendFatalError(response, the.getMessage());
		}
	}


	private boolean checkSupportedMediaTypes(HttpServletResponse oResp, MimeType resMediaType)
	throws IOException
	{
		String notImplemented = "NOT_IMPLEMENTED(code=201): Resolution media type '" + resMediaType + "' is not supported by this proxy";

		if (resMediaType == null) {
			if (!supportRedirect) {
				sendResponse(oResp, HTTP_ERROR_CONTENT_TYPE, 
						"NOT_IMPLEMENTED(code=201): HTTP redirect resolution mechanism is not supported by this proxy", null);
				return false;
			}
			return true;
		}

		if (!resMediaType.isValidXriResMediaType()) {
			sendResponse(oResp, HTTP_ERROR_CONTENT_TYPE, "INVALID_RESOLUTION_MEDAIA_TYPE(code=212): Unknown media type '" + resMediaType + "'", null);
			return false;
		}

		// must be one of the types
		if (resMediaType.isType(MimeType.XRDS_XML) && !supportXRDS) {
			sendResponse(oResp, HTTP_ERROR_CONTENT_TYPE, notImplemented, null);
			return false;
		}
		else if (resMediaType.isType(MimeType.XRD_XML) && !supportXRD) {
			sendResponse(oResp, HTTP_ERROR_CONTENT_TYPE, notImplemented, null);
			return false;
		}
		else if (resMediaType.isType(MimeType.URI_LIST) && !supportURIList) {
			sendResponse(oResp, HTTP_ERROR_CONTENT_TYPE, notImplemented, null);
			return false;
		}
		return true;
	}


	/**
	 * Checks to see if auto-correction is necessary. If so, return the auto-corrected string.
	 * @param s
	 * @return <code>null</code> if no auto-correction needs to be done, or the auto-corrected string otherwise.
	 */
	private String checkAutoCorrect(String s)
	{
		int i = 0;
		if (s.length() > 6 && s.substring(0, 6).equalsIgnoreCase("xri://"))
			i = 6;
		i = s.indexOf('/', i);
		if (i == -1)
			return null; // no path, ok

		String path = s.substring(i+1);
		String pathQuoted = quotePath(path);
		if (pathQuoted.equals(path))
			return null; // no need to quote

		// combine quoted path
		String newXRI = s.substring(0, i+1) + pathQuoted;
		return newXRI;
	}


	/**
	 * Process a request for proxied resolution
	 */
	private void processProxyRequest(
			String qxri, 
			MimeType resMediaType, 
			String serviceType, 
			String serviceMediaType,
			HttpServletRequest request, 
			HttpServletResponse response, 
			boolean isDumbBrowser)
	throws IOException
	{
		log.trace("processProxyRequest - enter");

		// check if we support the media types
		if (!checkSupportedMediaTypes(response, resMediaType)) {
			log.trace("processProxyRequest - checkSupportedMediaTypes returned false, returning.");
			return;
		}

		// Toss if something is amiss
		if ((qxri == null) || (qxri.length() == 0)) {
			log.debug("processProxyRequest - sXRI is null or empty");
			sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "INVALID_QXRI(code=211): null or empty", null);
			return;
		}

		XRI oXRI = null;
		try {
			oXRI = XRI.fromURINormalForm(qxri);
		}
		catch (XRIParseException oEx) {
			// log and send a 404
			log.warn("Error constructing XRI: " + qxri + ", " + oEx);
			sendResponse(response,HTTP_ERROR_CONTENT_TYPE, "INVALID_QXRI(code=211): "+oEx.getMessage(),null);
			return;
		}


		// defaults if resolution media type is null
		TrustType trustType = new TrustType();
		boolean refs = true;
		boolean sep = true;

		String tempStr = null;
		if(resMediaType != null) {
			tempStr = resMediaType.getParam(MimeType.PARAM_REFS);
			if(tempStr != null && tempStr.equals("false"))
				refs = false;

			tempStr = resMediaType.getParam(MimeType.PARAM_SEP);
			if(tempStr != null && tempStr.equals("false"))
				sep = false;

			tempStr = resMediaType.getParam(MimeType.PARAM_TRUST);
			if (tempStr != null) {
				try {
					trustType.setType(tempStr);
				}
				catch (IllegalTrustTypeException e) {
					sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "INVALID_RESOLUTION_MEDAIA_TYPE(code=212): " + resMediaType, null);
					return;
				}
			}
		}

		// set the request preferences on the resolver before querying.
		XRDS xrds = null;
		XRD xrd = null;
		try
		{
			if (sep) {
				if (resMediaType == null) {
					// see section 7.6 for special redirect rule
					ArrayList uris = resolver.resolveSEPToURIList(oXRI.toString(), trustType, serviceType, serviceMediaType, refs);
					if (uris == null || uris.size() == 0) {
						sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "SEP_NOT_FOUND(code=241): no url found", null);
						return;
					}
					String s = (String) uris.get(0);

					log.trace("Sending redirect to '" + s + "'");

					response.sendRedirect(s);
				}
				else if (resMediaType.isType(MimeType.URI_LIST)) {
					String  text = resolver.resolveSEPToTextURIList(oXRI.toString(), trustType, serviceType, serviceMediaType, refs);
					if (text.length() <= 0)
						sendResponse(response, HTTP_ERROR_CONTENT_TYPE, "SEP_NOT_FOUND(code=241): no url found", null);
					else
						sendResponse(response, resMediaType.getType(), text, null);
				}
				else if (resMediaType.isType(MimeType.XRDS_XML)) {
					xrds = resolver.resolveSEPToXRDS(oXRI, trustType, serviceType, serviceMediaType, refs);
					sendResponse(response, isDumbBrowser, resMediaType.getType(), xrds.toString(), trustType);
				}
				else if (resMediaType.isType(MimeType.XRD_XML)) {
					xrd = resolver.resolveSEPToXRD(oXRI, trustType, serviceType, serviceMediaType, refs);
					sendResponse(response, isDumbBrowser, resMediaType.getType(), xrd.toResultString(), trustType);
				}
				else {
					// else - we should have taken care of it in checkSupportedMediaTypes
					log.error("processProxyRequest - should not reach here (sep=true)");
				}
			}
			else {
				//// authority resolution only
				if(resMediaType == null) {
					resMediaType = new MimeType(MimeType.XRDS_XML);
				}

				if (resMediaType.isType(MimeType.XRDS_XML)) {
					xrds = resolver.resolveAuthToXRDS(oXRI, trustType, refs);
					sendResponse(response, isDumbBrowser, resMediaType.getType(), xrds.toString(), trustType);
					return;
				}
				else if (resMediaType.isType(MimeType.XRD_XML)) {
					xrd = resolver.resolveAuthToXRD(oXRI, trustType, refs);
					sendResponse(response, isDumbBrowser, resMediaType.getType(), xrd.toString(), trustType);
				}
				else if (resMediaType.isType(MimeType.URI_LIST)) {
					// ignore (must do SEP when text/uri-list is specified)
					log.warn("text/uri-list given but does not want to do service selection");
					// TODO: do we return an error?
				}
				else {
					// else - we should have taken care of it in checkSupportedMediaTypes
					log.error("processProxyRequest - should not reach here (sep=false)");
				}
			}
			return;
		}
		catch (PartialResolutionException pre) {
			log.info("processProxyRequest - caught PartialResolutionException");
			xrds = pre.getPartialXRDS();
			sendPartialResponse(request, response, isDumbBrowser, resMediaType, xrds, trustType);
		}
	}





	/**
	 * auto correction of input path parameter
	 * @param path
	 * @return
	 */
	private String quotePath (String path) {
		if (path == null || path.length() == 0)
			return path;

		char first = path.charAt(0);
		switch (first)
		{
		case '+':
		case '=':
		case '@':
		case '$':
		case '!':
		{
			// starts with GCS character, we need to parenthesize
			int i = path.indexOf('/');
			if (i == -1)
				// quote entire path since there is only one path segment
				return "(" + path + ")";
			else
				// insert the closing parenthesis right before the '/'
				return "(" + path.substring(0, i) + ")" + path.substring(i);
		}
		}
		return path;
	}


	protected String buildAbsoluteURL(HttpServletRequest request, String relPath)
	{
		StringBuffer sb = new StringBuffer();
		sb.append(request.getContextPath());

		log.trace("context='" + request.getContextPath() + "'");
		log.trace("path='" + request.getServletPath() + "'");

		log.trace("sb='" + sb + "'");

		int len = sb.length();
		if (len > 1 && sb.charAt(len-1) != '/')
			sb.append('/');

		log.trace("sb2='" + sb + "'");

		sb.append(request.getServletPath());
		len = sb.length();
		if (len > 1 && sb.charAt(len-1) != '/')
			sb.append('/');

		if (sb.length() == 0)
			sb.append('/');

		log.trace("sb3='" + sb + "'");

		sb.append(relPath);
		if (request.getQueryString() != null) {
			sb.append('?');
			sb.append(request.getQueryString());
		}
		log.trace("sb4='" + sb + "'");
		return sb.toString();
	}



	/**
	 * This method is called when an unknown error is encountered.
	 * This simply outputs the error message in text/plain.
	 * 
	 * @param resp
	 * @param errorMessage
	 */
	protected void sendFatalError(HttpServletResponse resp, String errorMessage){
		resp.setStatus(HttpServletResponse.SC_OK);
		resp.setContentType(HTTP_ERROR_CONTENT_TYPE);
		try {
			ServletOutputStream out = resp.getOutputStream();
			out.println("The proxy has encountered a fatal error. Internal error message follows: ");
			out.println(errorMessage);
		}
		catch (IOException e) {
		}
	}



	protected void sendPartialResponse(HttpServletRequest request, HttpServletResponse response, boolean isDumbBrowser, MimeType resMediaType, XRDS partialXRDS, TrustType trustType)
	throws IOException
	{
		log.trace("sendPartialResponse(dumbBrowser=" + isDumbBrowser + ", partialXRDS=" + partialXRDS.toString());

		response.setStatus(HttpServletResponse.SC_OK);
		XRD xrd = partialXRDS.getFinalXRD();
		String errMsg = "";
		if (xrd != null) {
			Status stat = xrd.getStatus();
			errMsg = "Error code: " + stat.getCode() + " - " + stat.getText();
		}

		if (resMediaType == null) {
			sendResponse(response, HTTP_ERROR_CONTENT_TYPE, errMsg, null);
		}
		else if (resMediaType.isType(MimeType.URI_LIST)) {
			errMsg = "# " + errMsg;
			sendResponse(response, HTTP_ERROR_CONTENT_TYPE, errMsg, trustType);
		}
		else if (resMediaType.isType(MimeType.XRDS_XML)) {
			sendResponse(response, isDumbBrowser, resMediaType.getType(), partialXRDS.toString(), trustType);
		}
		else if (resMediaType.isType(MimeType.XRD_XML)) {
			sendResponse(response, isDumbBrowser, resMediaType.getType(), xrd.toString(), trustType);
		}
		else {
			// else - we should have taken care of it in checkSupportedMediaTypes
			log.error("processProxyRequest - should not reach here (exception)");
		}
	}


	/**
	 * Send a HTTP response to the client.
	 * This method is designed to be overridden by subclasses for user-friendly
	 * display of errors.
	 * 
	 * @param isDumbBrowser flag to indicate if the client was detected to be a browser
	 */
	protected void sendResponse(
			HttpServletResponse response,
			boolean isDumbBrowser,
			String contentType,
			String result,
			TrustType trustType)
	throws IOException
	{
		if (isDumbBrowser)
			sendResponse(response, HTTP_XML_CONTENT_TYPE, result, trustType);
		else
			sendResponse(response, contentType, result, trustType);
	}

	/**
	 * Send a HTTP response to the client.
	 */
	protected void sendResponse(
			HttpServletResponse response, 
			String contentType, 
			String result, 
			TrustType trustType)
	throws IOException
	{
		sendResponse(response, HttpServletResponse.SC_OK, contentType, result, trustType);
	}


	/**
	 * Send a HTTP response with the specified status code to the client.
	 * 
	 * @param response
	 * @param statusCode
	 * @param contentType
	 * @param result
	 * @param trustType
	 * @throws IOException
	 */
	protected void sendResponse(
			HttpServletResponse response, 
			int statusCode,
			String contentType, 
			String result, 
			TrustType trustType)
	throws IOException
	{
		response.setStatus(statusCode);
		if (trustType != null)
			contentType += ";" + trustType.getParameterPair();
		response.setContentType(contentType);
		response.getOutputStream().print(result);
	}


	protected class QueryParams {
		String xrdR = null;
		String xrdT = null;
		String xrdM = null;
		String opaque = null;
	}

	protected QueryParams parseQuery(HttpServletRequest req)
	{
		QueryParams qp = new QueryParams();
		String queryString = req.getQueryString();
		String xrdR = null;
		String xrdT = null;
		String xrdM = null;
		StringBuffer opaque = new StringBuffer();

		if (queryString == null) {
			return qp;
		}

		log.trace("parseQuery() - queryString.length=" + queryString.length());

		int start = 0;
		while (start < queryString.length()) {
			log.trace("parseQuery() - start=" + start);
			String kvpair;
			int i = queryString.indexOf('&', start);
			log.trace("parseQuery() - i=" + i);
			if (i == -1) {
				kvpair = queryString.substring(start);
				start = queryString.length(); // done
			}
			else {
				kvpair = queryString.substring(start, i);
				start = i+1;
			}

			// parse kvpair
			int eq = kvpair.indexOf('=');
			String key, val;
			if (eq == -1) {
				key = kvpair;
				val = "";
			}
			else {
				key = kvpair.substring(0, eq);
				val = kvpair.substring(eq+1);
			}

			// see if we recognize the key
			if (xrdR == null && key.toLowerCase().equals(_XRD_R)) {
				log.trace("parseQuery() - xrdR=" + val);
				xrdR = val;
			}
			else if (xrdT == null && key.toLowerCase().equals(_XRD_T)) {
				log.trace("parseQuery() - xrdT=" + val);
				xrdT = val;
			}
			else if (xrdM == null && key.toLowerCase().equals(_XRD_M)) {
				log.trace("parseQuery() - xrdM=" + val);
				xrdM = val;
			}
			else {
				log.trace("parseQuery() - param=" + kvpair);
				// don't recognize it or we already have the parameter
				// just add it to the opaque string
				if (opaque.length() > 0)
					opaque.append('&');

				opaque.append(kvpair);
			}
		}

		qp.xrdR = URLUtils.decode(xrdR);
		qp.xrdT = URLUtils.decode(xrdT);
		qp.xrdM = URLUtils.decode(xrdM);
		qp.opaque = (opaque.length() > 0)? opaque.toString(): null;

		log.trace("parseQuery() - xrdR=" + qp.xrdR + ", xrdT=" + qp.xrdT + ", xrdM="+ qp.xrdM + ", opaque="+qp.opaque);
		return qp;
	}
	
	private void outputOpenIDServerHack(
			HttpServletResponse response,
			String server) throws IOException {
		
		PrintWriter writer = response.getWriter();
		
		writer.println("<html><head>\n");
		writer.println("<link rel=\"openid.server\" href=\"" + server + "\">");
		writer.println("</head>");
		writer.println("<body>");
		writer.println("Hi there. If you see this, some funny happened.");
		writer.println("</body>");
		writer.println("</html>");
	}
	
	private void outputOpenIDDelegateHack(
			HttpServletResponse response,
			String server,
			String identifier) throws IOException {
		
		PrintWriter writer = response.getWriter();
		
		writer.println("<html><head>\n");
		writer.println("<link rel=\"openid.server\" href=\"" + server + "\">");
		writer.println("<link rel=\"openid.delegate\" href=\"" + identifier + "\">");
		writer.println("</head>");
		writer.println("<body>");
		writer.println("Hi there. If you see this, some funny happened.");
		writer.println("</body>");
		writer.println("</html>");
	}
}
