package org.openxri.server.impl;

import java.util.Properties;

import org.openxri.GCSAuthority;
import org.openxri.IRIUtils;
import org.openxri.XRIAuthority;
import org.openxri.XRISegment;
import org.openxri.config.ComponentRegistry;
import org.openxri.config.PipelineRegistry;
import org.openxri.config.ServerConfig;
import org.openxri.config.impl.AbstractComponent;
import org.openxri.exceptions.ServerException;
import org.openxri.exceptions.ServerInternalException;
import org.openxri.exceptions.ServerNotFoundException;
import org.openxri.exceptions.StoreException;
import org.openxri.factories.ServerConfigFactory;
import org.openxri.pipeline.Pipeline;
import org.openxri.server.Server;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.store.StoreMountable;
import org.openxri.store.SubSegment;
import org.openxri.xml.ServerStatus;
import org.openxri.xml.Status;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;

/**
 * Provides a starting point for Server implementations. Subclasses are relieved from the task
 * of looking up authorities and retrieving information from the store. They only have to adjust
 * the individual XRDs as well as the final XRDS descriptor by implementing
 * initXRDS(), initXRD(), finishXRD() and finishXRDS()
 *
 * Subclasses are expected to add 'non-static' descriptor features that are not in the store, such as
 * Expires and Status elements.
 *
 * @author =peacekeeper
 * @see BasicServer
 */
public abstract class AbstractServer extends AbstractComponent implements Server {

	protected static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(AbstractServer.class.getName());

	protected Store store = null;

	public AbstractServer(Properties properties) {

		super(properties);
	}

	/**
	 * Initialize base-class attributes from configuration.
	 *
	 * @param oConfig The server configuration interface to initialize with
	 */
	public void init() {

		log.trace("init()");

		// we need the store

		ComponentRegistry componentRegistry = ServerConfigFactory.getSingleton().getComponentRegistry();

		this.store = (Store) componentRegistry.getComponent(Store.class);

		// done

		log.trace("Done.");
	}

	public XRDS lookupSelfDescribing(
			String namespace, 
			boolean signed)
	throws ServerException
	{
		log.trace("lookupSelfDescribing(" + namespace + ")");

		// find the authority object for this namespace

		String utf8Namespace;

		try {

			utf8Namespace = IRIUtils.IRItoXRI(IRIUtils.URItoIRI(namespace));
		} catch (Exception ex) {

			ServerException ex2 = new ServerInternalException("Unsupported encoding in namespace: " + ex.getMessage(), ex);
			log.warn("", ex2);
			throw ex2;
		}

		XRIAuthority authorityPath = new GCSAuthority(utf8Namespace);

		Authority namespaceAuthority;
		setParentQueryName(utf8Namespace);

		try {

			namespaceAuthority = this.store.localLookup(authorityPath);
		} catch (StoreException ex) {

			ServerException ex2 = new ServerInternalException("Error while finding namespace authority " + namespace + ".", ex);
			log.warn("", ex2);
			throw ex2;
		}

		if (namespaceAuthority == null) {

			ServerException ex = new ServerNotFoundException("Namespace authority " + namespace + " not found.");
			log.warn("", ex);
			throw ex;
		}

		// the big goal is to make an XRDS, consisting of one XRD

		XRDS xrds = new XRDS();
		boolean ret;

		// give subclasses a chance to init the XRDS before we begin

		ret = this.initXRDS(
				xrds, 
				namespace, 
				signed);

		// if a subclass returned true, it doesn't want us to do any more work

		if (ret == true) {

			log.debug("Subclass handled XRDS completely. Returning it without any more work.");
			return(xrds);
		}

		// generate an XRD for the namespace

		for (int i=0; i<1; i++) {

			try {

				// create a fresh XRD

				XRD xrd = new XRD();

				// give subclasses a chance to init the XRD

				ret = this.initXRD(
						xrd, 
						namespaceAuthority, 
						namespace, 
						signed);

				// if a subclass returned true, it doesn't want us to do any more work

				if (ret == true) {

					log.debug("Subclass handled XRD completely. Returning it without any more work.");
					xrds.add(xrd);
					break;
				}

				// check if the authority overrides the LOOKUP pipeline. if not, use the default.

				ServerConfig serverConfig = ServerConfigFactory.getSingleton();
				PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
				Pipeline lookupPipeline = null;

				if (this.store instanceof StoreAttributable) {

					StoreAttributable storeAttributable = (StoreAttributable) this.store;

					String pipelineName = storeAttributable.getAuthorityAttributes(namespaceAuthority).get(Pipeline.ATTRIBUTE_OVERRIDE_LOOKUP_PIPELINE);
					if (pipelineRegistry != null && pipelineName != null) lookupPipeline = pipelineRegistry.getPipelineByName(pipelineName);
				}

				if (pipelineRegistry != null && lookupPipeline == null) lookupPipeline = pipelineRegistry.getDefaultLookupPipeline();

				// execute LOOKUP pipeline

				xrd = lookupPipeline.execute(
						this.store, 
						xrd, 
						null, 
						null, 
						namespace, 
						namespaceAuthority, 
						false);

				// let our subclasses finish the XRD before we append it to the XRDS

				if (xrd != null) {

					this.finishXRD(
							xrd, 
							namespaceAuthority, 
							null, 
							null, 
							signed);
				}

				// if we were not able to get a XRD, return a "not found" XRD

				if (xrd == null) {

					xrds.add(makeNotFoundXrd(namespace));
					break;
				}

				// append XRD to the XRDS

				xrds.add(xrd);
			} catch (Exception ex) {

				log.error("", ex);
				xrds.add(makeExceptionXrd(namespace, ex));
				break;
			}
		}

		// let subclasses finish the XRDS before we send it out

		this.finishXRDS(
				xrds, 
				namespace, 
				signed);

		// done

		log.trace("Done.");
		return(xrds);
	}

	public XRDS lookupByNamespace(
			String namespace, 
			String query,
			boolean signed)
	throws ServerException
	{
		log.debug("lookupByNamespace(" + namespace + "," + query + ")");

		// find the authority object for this namespace

		String utf8Namespace;

		try {

			utf8Namespace = IRIUtils.IRItoXRI(IRIUtils.URItoIRI(namespace));
		} catch (Exception ex) {

			ServerException ex2 = new ServerInternalException("Unsupported encoding in namespace: " + ex.getMessage(), ex);
			log.warn("", ex2);
			throw ex2;
		}

		XRIAuthority authorityPath = new GCSAuthority(utf8Namespace);

		Authority parentAuthority;
		setParentQueryName(utf8Namespace);

		try {

			parentAuthority = this.store.localLookup(authorityPath);

			if (parentAuthority == null) throw new ServerNotFoundException("Namespace authority " + namespace + " not found.");
		} catch (StoreException ex) {

			ServerException ex2 = new ServerInternalException("Error while finding namespace authority " + namespace + ": " + ex.getMessage(), ex);
			log.warn("", ex2);
			throw ex2;
		}

		// the big goal is to make an XRDS, consisting of one or more XRDs

		XRDS xrds = new XRDS();
		boolean ret;

		// give subclasses a chance to init the XRDS before we begin

		ret = this.initXRDS(
				xrds, 
				query, 
				signed);

		// if a subclass returned true, it doesn't want us to do any more work

		if (ret == true) {

			log.debug("Subclass handled XRDS completely. Returning it without any more work.");
			return(xrds);
		}

		// generate XRDs for all subsegments in the request

		String utf8Query;

		try {

			utf8Query = IRIUtils.IRItoXRI(IRIUtils.URItoIRI(query));
		} catch (Exception ex) {

			ServerException ex2 = new ServerInternalException("Unsupported encoding in query: " + ex.getMessage(), ex);
			log.warn("", ex2);
			throw ex2;
		}

		XRISegment segment = new XRISegment(utf8Query);

		for (int i=0; i<segment.getNumSubSegments(); i++) {

			String subSegmentName = segment.getSubSegmentAt(i).toString();

			try {

				// create a fresh XRD

				XRD xrd = new XRD();

				// using the parent authority and subsegment name, find the subsegment and the authority we are describing now
				// we must be prepared to have null values here, since not all pipeline configurations
				// may use objects from the store

				SubSegment subSegment = null;
				Authority authority = null;

				if (parentAuthority != null) subSegment = this.store.findSubSegment(parentAuthority, subSegmentName);
				if (subSegment != null) authority = this.store.getSubSegmentAuthority(subSegment);

				// give subclasses a chance to init the XRD

				ret = this.initXRD(
						xrd, 
						parentAuthority, 
						subSegmentName, 
						signed);

				// if a subclass returned true, it doesn't want us to do any more work

				if (ret == true) {

					log.debug("Subclass handled XRD completely. Returning it without any more work.");
					xrds.add(xrd);
					continue;
				}

				// check if the parent authority overrides the LOOKUP pipeline. if not, use the default.

				ServerConfig serverConfig = ServerConfigFactory.getSingleton();
				PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
				Pipeline lookupPipeline = null;

				if (this.store instanceof StoreAttributable) {

					StoreAttributable storeAttributable = (StoreAttributable) this.store;

					String pipelineName = storeAttributable.getAuthorityAttributes(parentAuthority).get(Pipeline.ATTRIBUTE_OVERRIDE_LOOKUP_PIPELINE);
					if (pipelineRegistry != null && pipelineName != null) lookupPipeline = pipelineRegistry.getPipelineByName(pipelineName);
				}

				if (pipelineRegistry != null && lookupPipeline == null) lookupPipeline = pipelineRegistry.getDefaultLookupPipeline();

				// execute LOOKUP pipeline

				xrd = lookupPipeline.execute(
						this.store, 
						xrd, 
						segment, 
						parentAuthority, 
						subSegmentName, 
						authority, 
						false);

				// let our subclasses finish the XRD before we append it to the XRDS

				if (xrd != null) {

					this.finishXRD(
						xrd, 
						parentAuthority, 
						subSegment, 
						authority, 
						signed);
				}

				// if we were not able to get a XRD, that's ok if we found at least one before
				// (we are not necessarily authoritative for the whole query)

				if (xrd == null) {

					if (i > 0) {

						break;
					} else {

						xrds.add(makeNotFoundXrd(subSegmentName));
						break;
					}
				}

				// authority becomes the parent authority for the next subsegment

				parentAuthority = authority;
				if (subSegment != null) setParentQueryName(subSegment.getName());

				// append XRD to the XRDS

				xrds.add(xrd);
			} catch (Exception ex) {

				log.error("", ex);
				xrds.add(makeExceptionXrd(subSegmentName, ex));
				break;
			}
		}

		// let subclasses finish the XRDS before we send it out

		this.finishXRDS(
				xrds, 
				query, 
				signed);

		// done

		log.debug("Done.");
		return(xrds);
	}


	public XRDS lookupByPath(
			String path, 
			boolean signed)
	throws ServerException
	{
		log.debug("lookupByPath(" + path + ")");

		// the big goal is to make an XRDS, consisting of one or more XRDs

		XRDS xrds = new XRDS();
		boolean ret;

		// give subclasses a chance to init the XRDS before we begin

		setParentQueryName(path);

		ret = this.initXRDS(
				xrds, 
				null, 
				signed);

		// if a subclass returned true, it doesn't want us to do any more work

		if (ret == true) {

			log.debug("Subclass handled XRDS completely. Returning it without any more work.");
			return(xrds);
		}

		// find all mounted authorities for this path

		Authority[] authorities = new Authority[0];

		if (this.store instanceof StoreMountable) {

			try {

				StoreMountable storeMountable = (StoreMountable) this.store;

				authorities = storeMountable.listAuthoritiesByPath(path);
			} catch (StoreException ex) {

				ServerException ex2 = new ServerInternalException("Error while listing mounted authorities.", ex);
				log.warn("", ex2);
				throw ex2;
			}
		}

		// add XRDs of all mounted authorities

		for (int i=0; i<authorities.length; i++) {

			String subSegmentName = path;

			try {

				// create a fresh XRD

				XRD xrd = new XRD();

				// get the current authority from the list

				Authority authority = authorities[i];

				// give subclasses a chance to init the XRD

				ret = this.initXRD(
						xrd, 
						null, 
						subSegmentName, 
						signed);

				// if a subclass returned true, it doesn't want us to do any more work

				if (ret == true) {

					log.debug("Subclass handled XRD completely. Returning it without any more work.");
					xrds.add(xrd);
					continue;
				}

				// check if the authority overrides the LOOKUP pipeline. if not, use the default.

				ServerConfig serverConfig = ServerConfigFactory.getSingleton();
				PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
				Pipeline lookupPipeline = null;

				if (this.store instanceof StoreAttributable) {

					StoreAttributable storeAttributable = (StoreAttributable) this.store;

					String pipelineName = storeAttributable.getAuthorityAttributes(authority).get(Pipeline.ATTRIBUTE_OVERRIDE_LOOKUP_PIPELINE);
					if (pipelineRegistry != null && pipelineName != null) lookupPipeline = pipelineRegistry.getPipelineByName(pipelineName);
				}

				if (pipelineRegistry != null && lookupPipeline == null) lookupPipeline = pipelineRegistry.getDefaultLookupPipeline();

				// execute LOOKUP pipeline

				xrd = lookupPipeline.execute(
						this.store, 
						xrd, 
						null, 
						null, 
						subSegmentName, 
						authority, 
						false);

				// let our subclasses finish the XRD before we append it to the XRDS

				if (xrd != null) {

					this.finishXRD(
						xrd, 
						null, 
						null, 
						authority, 
						signed);
				}

				// if we were not able to get a XRD, return an error

				if (xrd == null) {

					xrds.add(makeNotFoundXrd(subSegmentName));
					break;
				}

				// append XRD to the XRDS

				xrds.add(xrd);
			} catch (Exception ex) {

				log.error("", ex);
				xrds.add(makeExceptionXrd(subSegmentName, ex));
				break;
			}
		}

		// let subclasses finish the XRDS before we send it out

		this.finishXRDS(
				xrds, 
				null, 
				signed);

		// done

		log.debug("Done.");
		return(xrds);
	}

	/*
	 * Methods for creating a "not found" and a "exception" XRD.
	 */

	public static XRD makeNotFoundXrd(String subSegmentName) {

		XRD xrd = new XRD();

		xrd.setQuery(subSegmentName);
		xrd.setStatus(new Status(Status.QUERY_NOT_FOUND, "No descriptor found for this query."));
		xrd.setServerStatus(new ServerStatus(Status.QUERY_NOT_FOUND, "No descriptor found for this query."));

		return(xrd);
	}

	public static XRD makeExceptionXrd(String subSegmentName, Exception ex) {

		XRD xrd = new XRD();

		String text = ex.getMessage();
		if (text == null || text.trim().equals("")) text = "An internal error occurred: " + ex.getClass().getName();

		xrd.setQuery(subSegmentName);
		xrd.setStatus(new Status(Status.TEMPORARY_FAIL, text));
		xrd.setServerStatus(new ServerStatus(Status.TEMPORARY_FAIL, text));

		return(xrd);
	}

	/*
	 * Methods for our subclasses to init/finish the XRDs and the overall XRDS
	 */

	/**
	 * Called before the server starts processing the request.
	 * @param xrds - A blank XRDS which we will complete.
	 * @param query - The original XRI segment for which the request was made.
	 * @param signed - Whether we are expected to return a signed descriptor.
	 * @return If true, the AbstractServer will not add anything to the XRDS and send it out immediately.
	 * @throws ServerInternalException
	 */
	protected abstract boolean initXRDS(
			XRDS xrds, 
			String query, 
			boolean signed)
	throws ServerInternalException;

	/**
	 * Called before the XRDS is sent out by the server.
	 * @param xrds - The XRDS filled with all XRDs.
	 * @param query - The original XRI segment for which the request was made.
	 * @param signed - Whether we are expected to return a signed descriptor.
	 * @throws ServerInternalException
	 */
	protected abstract void finishXRDS(
			XRDS xrds, 
			String query, 
			boolean signed)
	throws ServerInternalException;

	/**
	 * Called before anything is added to a fresh XRD.
	 * @param xrd - A blank XRD which we will fill.
	 * @param parent - The authority describing this XRD.
	 * @param subSegmentName - The name of the subsegment to be resolved.
	 * @param signed - Whether we are expected to return a signed descriptor.
	 * @return If true, the AbstractServer will append the XRD to the XRDS and return it without any further processing.
	 * @throws ServerInternalException
	 */
	protected abstract boolean initXRD(
			XRD xrd, 
			Authority parent, 
			String subSegmentName, 
			boolean signed)
	throws ServerInternalException;

	/**
	 * Called before the server adds the XRD to the XRDS and moves on to the next authority.
	 * 
	 * This method should be prepared to handle a null XRD.
	 * 
	 * @param xrd - A filled XRD ready to be appended to the XRDS.
	 * @param parent - The authority describing this XRD.
	 * @param subSegment - The subsegment leading to this XRD.
	 * @param authority - The authority described by this XRD.
	 * @param signed - Whether we are expected to return a signed descriptor.
	 * @throws ServerInternalException
	 */
	protected abstract void finishXRD(
			XRD xrd, 
			Authority parent, 
			SubSegment subSegment, 
			Authority authority,
			boolean signed)
	throws ServerInternalException;

	/**
	 * This is overriden by the TrustedServer which uses the parent subsegment name
	 * for generating SAML assertions.
	 * @param subSegment The name of the subsegment that was resolved in the previous step.
	 */
	protected void setParentQueryName(String subSegment) {

	}
}
