package org.openxri.server.impl;

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xml.security.keys.KeyInfo;
import org.apache.xml.security.keys.content.X509Data;
import org.openxri.config.ServerConfig;
import org.openxri.exceptions.ServerInternalException;
import org.openxri.factories.ServerConfigFactory;
import org.openxri.saml.Assertion;
import org.openxri.saml.Attribute;
import org.openxri.saml.AttributeStatement;
import org.openxri.saml.Conditions;
import org.openxri.saml.NameID;
import org.openxri.saml.Subject;
import org.openxri.store.Authority;
import org.openxri.store.SubSegment;
import org.openxri.util.DOMUtils;
import org.openxri.xml.Tags;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;
import org.w3c.dom.Document;

/**
 * An extension to BasicServer that adds trust information to the XRD before it is sent out.
 * 
 * TODO: Fix everything and implement trusted resolution correctly.
 * 
 * @author =peacekeeper
 * @author Peter Williams (home_pw@msn.com)
 */
public class TrustedServer extends BasicServer {

	protected static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(TrustedServer.class.getName());

	public static final String CONFIG_CERTIFICATE_LOCATION = "certificate.location";
	public static final String CONFIG_PRIVATE_KEY_LOCATION = "private.key.location";
	public static final String CONFIG_SAML_ISSUER = "saml.issuer";

	public static final String CONFIG_CERTIFICATE_LOCATION_DEFAULT = "/WEB-INF/issuercert.pem"; // from deployed war file...
	public static final String CONFIG_PRIVATE_KEY_LOCATION_DEFAULT = "/WEB-INF/issuerkey.p8";
	public static final String CONFIG_SAML_ISSUER_DEFAULT = "!0";

	private String samlKeyInfo;
	private String keyLocation;
	private String certificateLocation;
	private String samlIssuer;

	private X509Certificate[] cachedSamlCertChain;
	private PrivateKey cachedSamlPrivateKey;

	protected String parentQueryName;

	public TrustedServer(Properties properties) {

		super(properties);
	}

	@Override
	public void init() {

		super.init();

		ServerConfig serverConfig = ServerConfigFactory.getSingleton();

		this.keyLocation = serverConfig.getRealPath(this.properties.getProperty("private.key.location", CONFIG_PRIVATE_KEY_LOCATION_DEFAULT));
		this.certificateLocation = serverConfig.getRealPath(this.properties.getProperty("certificate.location", CONFIG_CERTIFICATE_LOCATION_DEFAULT));
		this.samlIssuer = this.properties.getProperty("saml.issuer", CONFIG_SAML_ISSUER_DEFAULT);

		// configure key and certificate for trusted resolution

		try {

			X509Certificate[] samlCertChain = this.getCertificateChain();

			Document oDoc = new DocumentImpl();
			KeyInfo oKeyInfo = new KeyInfo(oDoc);
			X509Data oData = new X509Data(oDoc);
			oData.addCertificate(samlCertChain[samlCertChain.length - 1]);
			oKeyInfo.add(oData);
			this.samlKeyInfo = DOMUtils.toString(oKeyInfo.getElement(), false, true);
		} catch (Exception oEx) {

			log.warn("Unable to create TrustedServer KeyInfo object for local authorities");
		}
	}

	@Override
	protected boolean initXRDS(
			XRDS xrds, 
			String query, 
			boolean bSigned) 
	throws ServerInternalException {

		// bail early if trusted res is requested and we are not configured for it

		if (bSigned && ((this.getCertificateChain() == null) || (this.getPrivateKey() == null))) {

			ServerInternalException ex = new ServerInternalException("Server not configured for trusted resolution");
			log.warn(ex);
			throw(ex);
		}

		return(super.initXRDS(xrds, query, bSigned));
	}

	@Override
	protected boolean initXRD(
			XRD xrd,
			Authority parentAuthority,
			String subSegmentName,
			boolean signed)
	throws ServerInternalException {

		// bail early if trusted res is requested and we are not configured for it

		if (signed && ((this.getCertificateChain() == null) || (this.getPrivateKey() == null))) {

			ServerInternalException ex = new ServerInternalException("Server not configured for trusted resolution");
			log.warn(ex);
			throw(ex);
		}
		return(super.initXRD(xrd,parentAuthority,subSegmentName,signed));
	}

	@Override
	protected void finishXRD(
			XRD xrd, 
			Authority parentAuthority, 
			SubSegment subSegment,
			Authority authority,
			boolean bSigned) 
	throws ServerInternalException {

		super.finishXRD(xrd, parentAuthority, subSegment, authority, bSigned);

		// generate an ID attribute for the XRD

		xrd.genXmlID();

		// add trust information to the XRD

		if ((bSigned) && (this.getCertificateChain() != null) && (this.getPrivateKey()!= null)) {

			try {

				Assertion oAssertion = createAssertion(
						xrd.getXmlID(), 
						xrd.getProviderID(),
						((subSegment == null) ? parentQueryName : subSegment.getName()));
				xrd.setSAMLAssertion(oAssertion);
				xrd.sign(this.getPrivateKey());
				xrd.setDOM(xrd.getDOM()); // persist saml sig in elem to type val
				{
					String foo = xrd.serializeDOM(false, true);
					log.debug("signed XRD is " + foo);
				}
				xrd.verifySignature(this.getCertificateChain()[0].getPublicKey());

			} catch (Exception oEx) {

				log.warn("Unable to saml sign descriptor: " + oEx);
			}
		}
	}

	/*
	 ****************************************************************************
	 * createAssertion()
	 ****************************************************************************
	 */ /**
	 * Generates an assertion that can be attached to the descriptor according
	 * to the rules of XRI trusted resolution.
	 * @param sXMLID - The id attribute of the descriptor
	 * @param sParentID - The authority id of the descriptor
	 * @param sResolved - The resolved subsegment
	 * @return Assertion that is SAML 2.0 compliant and has the basic information
	 * for XRI trusted resolution filled in.
	 */
	protected Assertion createAssertion(
			String sXMLID, 
			String sParentID, 
			String sResolved)
	{
		log.debug("createAssertion(" + sXMLID + "," + sParentID + "," + sResolved + ")");

		// create the SAMLSubject
		Subject oSubject = new Subject();
		NameID oName = new NameID(Tags.TAG_NAMEID);
		oName.setNameQualifier(sParentID);
		oName.setValue(sResolved);
		oSubject.setNameID(oName);

		// create the SAML Attribute Statement
		AttributeStatement oAttrStatement = new AttributeStatement();
		Attribute oAttr = new Attribute();
		oAttr.setValue("#" + sXMLID);
		oAttr.setName(Tags.NS_XRD_V2);
		oAttrStatement.setAttribute(oAttr);

		// create the conditions
		Conditions oConditions = new Conditions();

		// now we can create the true SAML Assertion
		Assertion oXRIAssertion = new Assertion();
		oXRIAssertion.setIssueInstant(DOMUtils.toXMLDateTime(new Date()));
		NameID oIssuer = new NameID(Tags.TAG_ISSUER);
		oIssuer.setValue(this.samlIssuer);
		oXRIAssertion.setIssuer(oIssuer);
		oXRIAssertion.setSubject(oSubject);
		oXRIAssertion.setAttrStatement(oAttrStatement);
		oXRIAssertion.setConditions(oConditions);

		log.debug("Done.");
		return oXRIAssertion;

	} // createAssertion()

	/**
	 * This method will read a certificate chain from a file and return it.
	 * The certificate chain is also stored in a member field for subsequent calls to this method.
	 * Subclasses can override this to get the certificate chain in some other way.
	 */
	protected X509Certificate[] getCertificateChain() throws ServerInternalException {

		if (this.cachedSamlCertChain == null) {

			FileInputStream certStream = null;

			Collection<?> certs = null;

			try {

				certStream = new FileInputStream(this.certificateLocation);

				CertificateFactory factory =
					CertificateFactory.getInstance("X.509");

				certs = factory.generateCertificates(certStream);

				this.cachedSamlCertChain = new X509Certificate[certs.size()];

				Iterator<?> certIter = certs.iterator();

				for (int i = 0; certIter.hasNext(); i++) {

					this.cachedSamlCertChain[i] = (X509Certificate) certIter.next();
				}
			} catch (Exception e) {

				throw new ServerInternalException(
						"XRI Server Setup Error: Failed to read certificate chain: ", e);
			} finally {

				if (certStream != null) {

					try {

						certStream.close();
					} catch (Exception e) {}
				}
			}
		}

		return this.cachedSamlCertChain;
	}

	/**
	 * This method will read a private key from a file and return it.
	 * The private key is also stored in a member field for subsequent calls to this method.
	 * Subclasses can override this to get the private key in some other way.
	 */
	protected PrivateKey getPrivateKey() throws ServerInternalException {

		if (this.cachedSamlPrivateKey == null) {

			try {

				FileInputStream keyStream = new FileInputStream(this.keyLocation);

				ByteArrayOutputStream keyBytes = new ByteArrayOutputStream();

				int availableBytes = keyStream.available();

				while (availableBytes > 0) {

					byte[] bytes = new byte[availableBytes];

					keyStream.read(bytes);

					keyBytes.write(bytes);

					availableBytes = keyStream.available();
				}

				byte[] derKey = keyBytes.toByteArray();

				PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(derKey);

				KeyFactory keyFactory = KeyFactory.getInstance("RSA");

				this.cachedSamlPrivateKey = keyFactory.generatePrivate(keySpec);
			} catch (Exception e) {

				throw new ServerInternalException(
						"XRI Server Setup Error: Failed to read private key: ", e);
			}
		}

		return this.cachedSamlPrivateKey;
	}

	@Override
	protected void setParentQueryName(String subSegment) {

		this.parentQueryName = subSegment;
	}
}

