package org.openxri.servlet;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openxri.config.Component;
import org.openxri.config.ProxyConfig;
import org.openxri.config.impl.PropertiesProxyConfig;
import org.openxri.factories.ProxyConfigFactory;
import org.openxri.proxy.Proxy;
import org.openxri.proxy.ProxyException;

/**
 * Provides a servlet implementation for the XRI resolution protocol
 *
 * @author =wil
 * @author srinivasa.adapa@neustar.com
 */
public class ProxyServlet
extends HttpServlet
{
	protected static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog( ProxyServlet.class.getName());

	/**
	 * The class which will provide our configuration
	 */
	public static final String PROXY_CONFIG_CLASS = "proxy.config.class";
	public static final String DEFAULT_PROXY_CONFIG_CLASS = PropertiesProxyConfig.class.getName();

	private ProxyConfig proxyConfig;

	private Proxy moProxy;

	public void init() throws ServletException {

		// create ProxyConfig object

		ServletConfig servletConfig = this.getServletConfig();

		try {

			this.proxyConfig = ProxyConfigFactory.initSingleton(servletConfig);
		} catch (Exception ex) {

			throw new ServletException("Cannot initialize proxy configuration.", ex);
		}

		// now initialize the servlet

		try {
			
			init(proxyConfig);
		} catch (ProxyException ex) {
			
			throw new ServletException("Unable to initialize proxy.", ex);
		}
	}

	/**
	 * Initializes the servlet based on a ProxyConfig
	 */
	public void init(ProxyConfig oConfig) throws ProxyException
	{
		// try to load proxy implementation class

		ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
		String implementationName = this.proxyConfig.getProxyImplClass();
		Class<? extends Component> implementationClass;

		try {

			implementationClass = (Class<? extends Component>) classLoader.loadClass(implementationName);
		} catch (ClassNotFoundException ex) {

			log.error("Proxy implementation not found: " + implementationName, ex);
			throw new ProxyException("Proxy implementation not found: " + implementationName, ex);
		}

		// instantiate proxy implementation

		try {

			Class<?>[] constructorArgs = new Class<?>[] { ProxyConfig.class };
			Constructor<?> constructor = implementationClass.getConstructor(constructorArgs);
			this.moProxy = (Proxy) constructor.newInstance(new Object[] { this.proxyConfig });
		} catch (NoSuchMethodException ex) {

			log.error("Proxy " + implementationName + " does not have a constructor that takes a ProxyConfig object.", ex);
			throw new ProxyException("Proxy " + implementationName + " does not have a constructor that takes a ProxyConfig object.", ex);
		} catch (InstantiationException ex) {

			log.error("Cannot instantiate proxy " + implementationName + ".", ex);
			throw new ProxyException("Cannot instantiate proxy " + implementationName + ".", ex);
		} catch (IllegalAccessException ex) {

			log.error("Illegal access to constructor of proxy " + implementationName + ".", ex);
			throw new ProxyException("Illegal access to constructor of proxy " + implementationName + ".", ex);
		} catch (InvocationTargetException ex) {

			log.error("Cannot invoke constructor of proxy " + implementationName + ".", ex);
			throw new ProxyException("Cannot invoke constructor of proxy " + implementationName + ".", ex);
		} catch (Exception ex) {

			log.error("Cannot construct proxy " + implementationName + ".", ex);
			throw new ProxyException("Cannot construct proxy " + implementationName + ".", ex);
		}

		// init proxy

		this.moProxy.init();
	}

	public void destroy() {

		// shutdown proxy
		
		this.moProxy.shutdown();
	}
	
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		moProxy.process(request, response);
	}
}
