package org.openxri.servlet;

import java.io.IOException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openxri.config.ComponentRegistry;
import org.openxri.config.ServerConfig;
import org.openxri.factories.ServerConfigFactory;
import org.openxri.plugin.Plugin;
import org.openxri.resolve.MimeType;
import org.openxri.resolve.TrustType;
import org.openxri.resolve.exception.IllegalTrustTypeException;
import org.openxri.server.Server;
import org.openxri.urimapper.URIMapper;
import org.openxri.urimapper.URIMapperRequest;
import org.openxri.urimapper.URIMapperResult;
import org.openxri.xml.Tags;
import org.openxri.xml.XRDS;

/**
 * Provides a servlet implementation for the XRI resolution protocol
 * 
 * The XRIServlet works like this:
 * 1) During startup, construct an appropriate implementation of the ServerConfig interface,
 *    according to the init parameters in web.xml.
 * 2) When a request comes in, use the URIMapper to parse it.
 * 3) If the URIMapper does not know what to do with it, hand off the request to the Plugin.
 * 4) Let the Server implementation class build an XRDS for the request data extracted by the URIMapper.
 * 5) Send out the descriptor. 
 *
 * @author bgs
 * @author =chetan
 * @author =peacekeeper
 */
public class XRIServlet extends HttpServlet {

	private static final long serialVersionUID = 7240238001182896739L;

	protected static org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(XRIServlet.class.getName());

	/**
	 * The server configuration read from the XML file.
	 */
	protected ServerConfig serverConfig;

	/**
	 * The URI mapper that will tell us what to resolve
	 */
	protected URIMapper uriMapper;

	/**
	 * The server object that will handle the incoming resolution requests
	 */
	protected Server server;

	/**
	 * The plugin that will handle all requests not recognized by the URI mapper
	 */
	protected Plugin plugin;

	@Override
	public void init() throws ServletException {

		// create ServerConfig object

		ServletConfig servletConfig = this.getServletConfig();

		try {

			this.serverConfig = ServerConfigFactory.initSingleton(servletConfig);
		} catch (Exception ex) {

			throw new ServletException("Cannot initialize server configuration: " + ex.getMessage(), ex);
		}

		// now initialize the servlet

		init(this.serverConfig);
	}

	/**
	 * Initializes the servlet based on a ServerConfig
	 */
	public void init(ServerConfig serverConfig) {

		log.trace("init()");

		// get the components we need

		ComponentRegistry componentRegistry = serverConfig.getComponentRegistry();

		this.server = (Server) componentRegistry.getComponent(Server.class);
		this.uriMapper = (URIMapper) componentRegistry.getComponent(URIMapper.class);
		this.plugin = (Plugin) componentRegistry.getComponent(Plugin.class);

		log.trace("Done.");
	}

	@Override
	public void destroy() {

		log.trace("destroy()");

		log.trace("Done.");
	} 

	/**
	 * Service an incoming request.
	 */
	@Override
	public void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException {

		log.trace("doGet()");

		// check if we want trusted resolution

		boolean signed = checkSigned(request);

		// the goal is to get a complete XRDS document

		XRDS xrds = null;

		// let our URIMapper tell us what to do

		URIMapperResult data = this.uriMapper.parseRequest(new URIMapperRequest(request));

		String namespace = (data == null) ? null : data.getNamespace();
		String query = (data == null) ? null : data.getQuery();

		if (namespace != null && namespace.trim().equals("")) namespace = null;
		if (query != null && query.trim().equals("")) query = null;

		String path = request.getRequestURI().substring(request.getContextPath().length() + 1);

		// use the Server to get an XRDS

		try {

			// got a namespace but no query? that means the client wants a self-describing XRDS

			if (namespace != null && query == null) {

				// ask the Server to give us a self-describing XRDS for the namespace

				log.debug("Looking up self-describing descriptor.");

				xrds = this.server.lookupSelfDescribing(namespace, signed);
			}

			// got a namespace and a query? that means the client wants normal authority resolution

			if (namespace != null && query != null) {

				// ask the Server to give us an XRDS for the namespace and query

				log.debug("Looking up descriptor by namespace and query.");

				xrds = this.server.lookupByNamespace(namespace, query, signed);
			}

			// got neither a namespace nor a query? try getting an XRDS by path

			if (namespace == null && query == null) {

				// ask the Server to give us an XRDS for the path

				log.debug("Looking up descriptor by path.");

				xrds = this.server.lookupByPath(path, signed);
			}
		} catch (Exception ex) {

			log.warn("Internal server problem during resolution.", ex);
			response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, ex.getMessage());
			return;
		}

		// if we were not able to get an XRDS, let the plugin handle the request

		if (xrds == null || xrds.getNumXRD() < 1) {

			if (this.plugin == null) {

				log.warn("No plugin installed.");
				response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Invalid request.");
				return;
			}

			log.debug("Forwarding request to plugin.");

			boolean ret = this.plugin.processCustomRequest(request, response);

			// did the plugin handle the request?

			if (ret == true) {

				log.debug("Plugin successfully handled the request.");
				return;
			} else {

				log.warn("Plugin failed to handle the request.");
				response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Plugin failed to handle the request.");
				return;
			}
		}

		// send out the result

		String body = xrds.serializeDescriptorDOM(false, true);

		log.debug("Resolution successful. Sending descriptor.");
		response.setStatus(HttpServletResponse.SC_OK);
		
		if (xrds.getFinalXRD().getSAMLAssertion() != null)
			response.setContentType(Tags.CONTENT_TYPE_XRDS + ";saml=true;charset=UTF-8");
		else
			response.setContentType(Tags.CONTENT_TYPE_XRDS + ";charset=UTF-8");

		response.getOutputStream().write(body.getBytes("UTF-8"));
		response.flushBuffer();
	}

	/**
	 * This method can check whether the client requests SAML trusted resolution.
	 * 
	 * @param request The incoming HTTP request
	 * @return Whether SAML trusted resolution is requested.
	 */
	public static boolean checkSigned(HttpServletRequest request) {

		String sAccept = request.getHeader(Tags.HEADER_ACCEPT);

		if (sAccept == null) return false;

		MimeType requestedMimeType = MimeType.parse(sAccept);
		if (requestedMimeType == null) return false;

		if (! requestedMimeType.getType().equals(MimeType.XRDS_XML)) return false;
		
		String param = requestedMimeType.getParam(MimeType.PARAM_SAML);
		if (param == null) return false;

		return param.equals("true");
	}
}
