package org.openxri.store;

import org.openxri.config.Component;
import org.openxri.exceptions.StoreException;
import org.openxri.xml.XRD;

/**
 * @author =chetan
 * @author =peacekeeper
 */
public interface Store extends StoreLookup, Component {

	/**
	 * Creates a root subsegment (entry-point namespace to the server).
	 * @param name The name of the new root subsegment.
	 * @param xrd The XRD of the authority associated with the root subsegment.
	 */
	public SubSegment createRootSubSegment(
			String name,
			XRD xrd)
	throws StoreException;

	/**
	 * Creates an authority without associating it with a subsegment.
	 * You probably want to use registerSubsegment() instead, which creates both
	 * and authority and an associated subsegment.
	 * @param xrd An initial XRD that will be run through the default create pipeline and then associated with the new authority.
	 */
	public Authority createAuthority (
			XRD xrd) 
	throws StoreException;

	/**
	 * Registers a subsegment with a new authority.
	 * @param parentAuthority The authority to register under.
	 * @param subSegmentName The name of the subsegment to register.
	 * @param xrd An initial XRD that will be run through the default create pipeline and then associated with the new authority.
	 */
	public SubSegment registerSubsegment(
			Authority parentAuthority, 
			String subSegmentName, 
			XRD xrd)
	throws StoreException;

	/**
	 * Registers a new subsegment using an already existing authority.
	 * @param parentAuthority The authority id to register under.
	 * @param subSegmentName The name of the subsegment to register.
	 * @param authority The existing authority for the subsegment.
	 */
	public SubSegment registerSubsegment(
			Authority parentAuthority, 
			String subSegmentName, 
			Authority authority) 
	throws StoreException;

	/**
	 * Lists all of the root subsegments (namespaces) on the server.
	 */
	public SubSegment[] listRootSubSegments()
	throws StoreException;

	/**
	 * Releases the descriptor information associated with a registered
	 * subsegment.
	 * @param subSegment The subsegment to release.
	 */
	public void releaseSubSegment(
			SubSegment subSegment)
	throws StoreException;

	/**
	 * Deletes the root subsegment (namespace) associated with an authority.  Does
	 * not delete the authority itself.
	 * @param namespace The root subsegment (namespace) to delete.
	 */
	public void deleteRootSubSegment(
			String namespace)
	throws StoreException;

	/**
	 * Deletes an authority and any subsegments registered underneath it.  It does
	 * not delete any authorities registered for those subsegments.
	 * @param authority The authority to delete
	 */
	public void deleteAuthority(
			Authority authority)
	throws StoreException;
}
