package org.openxri.store;

import java.util.Map;

import org.openxri.exceptions.StoreException;


/**
 * Provides methods for associating attributes with subsegments and authorities. The attributes are
 * String key/value pairs and can include information such as registration date, username etc.
 * 
 * @author =peacekeeper
 */
public interface StoreAttributable {
	
	/**
	 * Associates attributes with the store.
	 * @param attributes The attributes.
	 */
	public void setStoreAttributes(Map<String, String> attributes) throws StoreException;
	
	/**
	 * Gets attributes from the store.
	 * @return The attributes, or null.
	 */
	public Map<String, String> getStoreAttributes() throws StoreException;
	
	/**
	 * Associates attributes with an authority.
	 * @param authority The authority.
	 * @param attributes The attributes.
	 */
	public void setAuthorityAttributes(Authority authority, Map<String, String> attributes) throws StoreException;

	/**
	 * Gets attributes from an authority.
	 * @param authority The authority.
	 * @return The attributes, or null.
	 */
	public Map<String, String> getAuthorityAttributes(Authority authority) throws StoreException;

	/**
	 * Associates an index with an authority.
	 * @param authority The authority.
	 * @param indx The index for the authority.
	 */
	public void setAuthorityIndex(Authority authority, String indx) throws StoreException;

	/**
	 * Gets the index of the authority.
	 * @param subSegment The authority.
	 */
	public String getAuthorityIndex(Authority authority) throws StoreException;

	/**
	 * Associates attributes with a subsegment.
	 * @param subSegment The subsegment.
	 * @param attributes The attributes.
	 */
	public void setSubSegmentAttributes(SubSegment subSegment, Map<String, String> attributes) throws StoreException;

	/**
	 * Gets attributes from a subsegment.
	 * @param subSegment The subsegment.
	 * @return The attributes, or null.
	 */
	public Map<String, String> getSubSegmentAttributes(SubSegment subSegment) throws StoreException;

	/**
	 * Associates an index with a subsegment.
	 * @param subSegment The subsegment.
	 * @param indx The index for the subsegment.
	 */
	public void setSubSegmentIndex(SubSegment subSegment, String indx) throws StoreException;

	/**
	 * Gets the index of the subsegment.
	 * @param subSegment The subsegment.
	 */
	public String getSubSegmentIndex(SubSegment subSegment) throws StoreException;

	/**
	 * Returns all authorities whose attributes satisfy certain conditions.
	 * @param selector An object that can select authorities based on their attributes.
	 * @return An array of authorities.
	 * @throws StoreException
	 */
	public Authority[] listAuthoritiesByAttributes(Selector selector) throws StoreException;

	/**
	 * Returns all subsegments whose attributes satisfy certain conditions.
	 * @param selector An object that can select authorities based on their attributes.
	 * @return An array of subsegments.
	 * @throws StoreException
	 */
	public SubSegment[] listSubSegmentsByAttributes(Selector selector) throws StoreException;

	/**
	 * Returns all authorities with a certain attribute key/value pair.
	 * @param key An attribute key.
	 * @param value An attribute value.
	 * @return An array of authorities.
	 * @throws StoreException
	 */
	public Authority[] listAuthoritiesByAttributeValue(String key, String value) throws StoreException;

	/**
	 * Returns all subsegments whose attributes satisfy certain conditions.
	 * @param key An attribute key.
	 * @param value An attribute value.
	 * @return An array of subsegments.
	 * @throws StoreException
	 */
	public SubSegment[] listSubSegmentsByAttributeValue(String key, String value) throws StoreException;

	/**
	 * Returns all authorities with a certain index.
	 * @param indx The index to look for.
	 * @return An array of authorities.
	 * @throws StoreException
	 */
	public Authority[] listAuthoritiesByIndex(String indx) throws StoreException;

	/**
	 * Returns all subsegments with a certain index.
	 * @param indx The index to look for.
	 * @return An array of subsegments.
	 * @throws StoreException
	 */
	public SubSegment[] listSubSegmentsByIndex(String indx) throws StoreException;

	/**
	 * An interface that can select subsegments and authorities by their attributes.
	 */
	public static interface Selector {

		/**
		 * @param attributes The attributes to be checked.
		 * @return True if the authority or subsegment will be selected.
		 */
		public boolean select(Map<String, String> attributes);
	}
}
