package org.openxri.store;

import org.openxri.exceptions.StoreException;


/**
 * Advanced methods for getting information about authorities and subsegments out of a store.
 * @author =peacekeeper
 */
public interface StoreBetterLookup extends StoreLookup {

	/**
	 * Returns all authorities.
	 * @return A list of authorities
	 * @throws StoreException
	 */
	public Authority[] listAuthorities() throws StoreException;

	/**
	 * Returns all subsegments.
	 * @return A list of subsegments
	 * @throws StoreException
	 */
	public SubSegment[] listSubSegments() throws StoreException;

	/**
	 * Gets all subsegments that lead to this authority.
	 * @param authority authority for which subsegments should be found
	 * @return An array of subsegments.
	 */
	public SubSegment[] getAuthoritySubSegments(
			Authority authority) throws StoreException;

	/**
	 * Gets all subsegments that have this authority as parent.
	 * @param parent authority for which child subsegments should be found
	 * @return An array of subsegments.
	 */
	public SubSegment[] getAuthorityChildSubSegments(
			Authority parent) throws StoreException;

	/**
	 * Gets all subsegments with a given parent authority and authority.
	 * All returned subsegments have the same parent authority and authority, and
	 * are therefore local synonyms.
	 * @param parent authority for which child subsegments should be found
	 * @param authority authority for which subsegments should be found
	 * @return An array of subsegments.
	 */
	public SubSegment[] getSynonymSubSegments(
			Authority parent,
			Authority authority) throws StoreException;
	
	/**
	 * Gets all full XRIs that lead to this authority.
	 * If it is a root authority, this is just the namespace, otherwise it's the
	 * namespace of the root authority plus names of subsegments. Usually just one result,
	 * unless there are aliases in the subsegment chain.
	 * 
	 * @param authority Authority for which all full names should be found
	 * @return An array of QXRIs.
	 */
	public String[] getAuthorityQxris(
			Authority authority) throws StoreException;

	/**
	 * Gets all full XRIs that lead to this authority.
	 * If it is a root authority, this is just the namespace, otherwise it's the
	 * namespace of the root authority plus names of subsegments. Usually just one result,
	 * unless there are aliases in the subsegment chain.
	 * 
	 * The reassignable and persistent parameters can be used to only read i-names or
	 * i-numbers from the store.
	 * 
	 * @param authority Authority for which all full names should be found
	 * @param reassignable Whether to consider reassignable subsegments when constructing XRIs. 
	 * @param persistent Whether to consider persistent subsegments when constructing XRIs. 
	 * @return An array of QXRIs.
	 */
	public String[] getAuthorityQxris(
			Authority authority,
			boolean reassignable,
			boolean persistent) throws StoreException;
}
