package org.openxri.store;

import org.openxri.exceptions.StoreException;

/**
 * Provides methods for mounting authorities at fixed paths.
 * 
 * If a request to that path is made, the authority's XRD will be put 
 * into an XRDS document and served to the client, after running the XRD
 * through the LOOKUP pipeline.
 * 
 * This is not used for XRI resolution, but may be used for other
 * XRDS or XRDS-Simple applications.
 * 
 * Authorities that are mounted at a path may still be used in XRI 
 * resolution, if there are subsegments that resolve to them.
 * 
 * More than one authority can be mounted at any given path. In this case,
 * the XRDS will contain each authority's XRD.
 * 
 * @author =peacekeeper
 */
public interface StoreMountable {

	/**
	 * Mounts an authority at a given path.
	 * @param path The path at which to mount the authority (null to unmount it).
	 * @param authority The authority to mount.
	 */
	public void setAuthorityPath(Authority authority, String path) throws StoreException;

	/**
	 * Returns the path at which the authority is mounted.
	 * @param authority The authority to check.
	 * @return The mount path of the authority, or null if the authority is not mounted.
	 */
	public String getAuthorityPath(Authority authority) throws StoreException;

	/**
	 * Returns all authorities mounted at a given path.
	 * @param path The mount path to look for.
	 * @return An array of authorities.
	 * @throws StoreException
	 */
	public Authority[] listAuthoritiesByPath(String path) throws StoreException;
}
