package org.openxri.store.impl.db;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Hibernate;
import org.hibernate.LockMode;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.openxri.XRIAuthority;
import org.openxri.config.PipelineRegistry;
import org.openxri.config.ServerConfig;
import org.openxri.exceptions.StoreDuplicateEntryException;
import org.openxri.exceptions.StoreException;
import org.openxri.exceptions.StoreInternalException;
import org.openxri.exceptions.StoreNotFoundException;
import org.openxri.factories.ServerConfigFactory;
import org.openxri.pipeline.Pipeline;
import org.openxri.store.Authority;
import org.openxri.store.Store;
import org.openxri.store.StoreAttributable;
import org.openxri.store.StoreBetterLookup;
import org.openxri.store.StoreEditable;
import org.openxri.store.StoreLookup;
import org.openxri.store.StoreMountable;
import org.openxri.store.StoreResettable;
import org.openxri.store.StoreStatistics;
import org.openxri.store.SubSegment;
import org.openxri.store.impl.AbstractStore;
import org.openxri.xml.XRD;

/**
 * A class using Hibernate for implementing the basic and several extended store operations. It expects
 * Hibernate parameters in the store arguments which should be present in the OpenXRI server configuration (e.g. in the
 * properties file).
 * 
 * @author =peacekeeper
 */
public class DatabaseStore 
extends
AbstractStore
implements
Store,
StoreLookup,
StoreBetterLookup,
StoreEditable,
StoreResettable,
StoreAttributable,
StoreMountable,
StoreStatistics {

	private static final Log log = LogFactory.getLog(DatabaseStore.class.getName());

	public static final int MAX_ALIAS_RECURSION = 30;

	protected Configuration configuration;
	protected SessionFactory sessionFactory;

	public DatabaseStore(Properties properties) {

		super(properties);
	}

	public synchronized void init() throws StoreException {

		log.trace("init()");

		try {

			// prepare Hibernate configuration

			this.configuration = new Configuration();

			this.configuration.setProperties(this.properties);

			this.configuration.addClass(org.openxri.store.impl.db.DbAuthority.class);
			this.configuration.addClass(org.openxri.store.impl.db.DbSubSegment.class);
			this.configuration.addClass(org.openxri.store.impl.db.DbStoreData.class);

			// create session factory

			this.initSessionFactory();
		} catch (Exception ex) {

			log.error("", ex);
			throw new StoreInternalException(ex, "Cannot initialize Hibernate");
		}

		log.trace("Done.");
	}

	private synchronized void initSessionFactory() {

		this.sessionFactory = this.configuration.buildSessionFactory();
	}

	public synchronized void close() {

		log.trace("close()");

		this.sessionFactory.close();
		this.sessionFactory = null;
	}

	public synchronized boolean isClosed() {

		return(this.sessionFactory == null || this.sessionFactory.isClosed());
	}

	/**
	 * Checks if the database connection is still alive;
	 * if not, try to reconnect, then throw exception.
	 * @return The database connection.
	 */
	public synchronized SessionFactory getSessionFactory() throws StoreException {

		if (this.sessionFactory != null && ! this.sessionFactory.isClosed()) return(this.sessionFactory);

		this.initSessionFactory();

		if (this.sessionFactory != null && ! this.sessionFactory.isClosed()) return(this.sessionFactory);

		throw new StoreInternalException("Database not available.");
	}

	/**
	 * Allow the connection to be changed externally.
	 * @param sessionFactory
	 */
	public synchronized void setSessionFactory(SessionFactory sessionFactory) {

		this.sessionFactory = sessionFactory;
	}

	/*
	 * Here comes the implementation for Store.
	 */

	public synchronized SubSegment[] listRootSubSegments() 
	throws StoreException {

		log.trace("listRootSubSegments()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// list all root subsegments

		List<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			subSegments = DbSubSegment.AllRoot(session);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot list root subsegments.");
		}

		// build result list

		SubSegment[] result = new ArrayList<SubSegment> (subSegments).toArray(new SubSegment[subSegments.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized Authority createAuthority (
			XRD xrd) 
	throws StoreException {

		log.trace("createAuthority()");

		// use the default CREATE pipeline.

		ServerConfig serverConfig = ServerConfigFactory.getSingleton();
		PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
		Pipeline createPipeline = null;

		if (pipelineRegistry != null) createPipeline = pipelineRegistry.getDefaultCreatePipeline();

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// create authority

		DbAuthority authority;

		try {

			session.beginTransaction();

			authority = new DbAuthority();

			session.save(authority);
			session.flush();
			session.refresh(authority);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// run XRD through the CREATE pipeline

		XRD newXRD;

		try {

			if (createPipeline != null) {

				newXRD = createPipeline.execute(this, xrd, null, null, null, authority, true);
			} else {

				newXRD = xrd;
			}
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot execute pipeline.");
		}

		// start database session

		session = this.getSessionFactory().getCurrentSession();

		// set XRD

		try {

			session.beginTransaction();

			reattach(session, authority);

			authority.setXrd(newXRD);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authority);
	}

	public synchronized SubSegment createRootSubSegment(
			String name,
			XRD xrd) 
	throws StoreException {

		log.trace("createRootSubSegment()");

		// use the default CREATE pipeline.

		ServerConfig serverConfig = ServerConfigFactory.getSingleton();
		PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
		Pipeline createPipeline = null;

		if (pipelineRegistry != null) createPipeline = pipelineRegistry.getDefaultCreatePipeline();

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// create root authority

		DbAuthority authority;

		try {

			session.beginTransaction();

			if (DbSubSegment.RootByName(session, name) != null) {

				throw new StoreDuplicateEntryException("Root subsegment exists already.");
			}

			authority = new DbAuthority();

			session.save(authority);
			session.flush();
			session.refresh(authority);
			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// run XRD through the CREATE pipeline

		XRD newXRD;

		try {

			if (createPipeline != null) {

				newXRD = createPipeline.execute(this, xrd, null, null, null, authority, true);
			} else {

				newXRD = xrd;
			}
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot execute pipeline.");
		}

		// start database session

		session = this.getSessionFactory().getCurrentSession();

		// set XRD

		try {

			session.beginTransaction();

			reattach(session, authority);

			authority.setXrd(newXRD);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// start database session

		session = this.getSessionFactory().getCurrentSession();

		// create root subsegment

		DbSubSegment subSegment;

		try {

			session.beginTransaction();

			reattach(session, authority);

			subSegment = new DbSubSegment();
			subSegment.setName(name);
			subSegment.setAuthority(authority);

			session.save(subSegment);
			session.flush();
			session.refresh(subSegment);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegment);
	}

	public synchronized void deleteRootSubSegment(
			String name) 
	throws StoreException {

		log.trace("deleteRootSubSegment()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// delete root subsegment

		DbSubSegment rootSubSegment;

		try {

			session.beginTransaction();

			rootSubSegment = DbSubSegment.RootByName(session, name);

			if (rootSubSegment == null) {

				throw new StoreNotFoundException("Root subsegment does not exist.");
			}

			session.delete(rootSubSegment);
			session.flush();
			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public synchronized void deleteAuthority(
			Authority authority) 
	throws StoreException {

		log.trace("deleteAuthority()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// delete authority

		try {

			session.beginTransaction();

			session.delete(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public synchronized SubSegment registerSubsegment (
			Authority parentAuthority, 
			String subSegmentName, 
			XRD xrd) 
	throws StoreException {

		log.trace("registerSubsegment(" + subSegmentName + ")");

		// check parent authority

		if (parentAuthority == null) throw new NullPointerException();
		if (! (parentAuthority instanceof DbAuthority)) throw new StoreNotFoundException("Parent authority is not from this store.");

		// check if the parent authority overrides the CREATE pipeline. if not, use the default.

		ServerConfig serverConfig = ServerConfigFactory.getSingleton();
		PipelineRegistry pipelineRegistry = (serverConfig == null) ? null : serverConfig.getPipelineRegistry();
		Pipeline createPipeline = null;

		String pipelineName = this.getAuthorityAttributes(parentAuthority).get(Pipeline.ATTRIBUTE_OVERRIDE_CREATE_PIPELINE);
		if (pipelineRegistry != null && pipelineName != null) createPipeline = pipelineRegistry.getPipelineByName(pipelineName);
		if (pipelineRegistry != null && createPipeline == null) createPipeline = pipelineRegistry.getDefaultCreatePipeline();

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// create authority

		DbAuthority authority;

		try {

			session.beginTransaction();

			reattach(session, parentAuthority);

			if (DbSubSegment.ByParentAndName(session, (DbAuthority) parentAuthority, subSegmentName) != null) {

				throw new StoreDuplicateEntryException("Subsegment exists already.");
			}

			authority = new DbAuthority();

			session.save(authority);
			session.flush();
			session.refresh(authority);
			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// run XRD through the CREATE pipeline

		XRD newXRD;

		try {

			if (createPipeline != null) {

				newXRD = createPipeline.execute(this, xrd, null, parentAuthority, subSegmentName, authority, true);
			} else {

				newXRD = xrd;
			}
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot execute pipeline.");
		}

		// start database session

		session = this.getSessionFactory().getCurrentSession();

		// set XRD

		try {

			session.beginTransaction();

			reattach(session, authority);

			authority.setXrd(newXRD);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// start database session

		session = this.getSessionFactory().getCurrentSession();

		// create subsegment with this authority

		DbSubSegment subSegment;

		try {

			session.beginTransaction();

			reattach(session, parentAuthority);
			reattach(session, authority);

			subSegment = new DbSubSegment();
			subSegment.setParent((DbAuthority)  parentAuthority);
			subSegment.setName(subSegmentName);
			subSegment.setAuthority(authority);

			((DbAuthority) parentAuthority).getChildren().add(subSegment);
			authority.getSubSegments().add(subSegment);

			session.save(subSegment);
			session.flush();
			session.refresh(subSegment);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegment);
	}

	public synchronized SubSegment registerSubsegment (
			Authority parentAuthority, 
			String subSegmentName, 
			Authority authority)
	throws StoreException {

		log.trace("registerSubsegment(" + subSegmentName + ")");

		// check authority and parent authority

		if (parentAuthority != null && (! (parentAuthority instanceof DbAuthority))) throw new StoreNotFoundException("Parent authority is not from this store.");
		if (authority != null && (! (authority instanceof DbAuthority))) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// create subsegment with this authority

		DbSubSegment subSegment;

		try {

			session.beginTransaction();

			reattach(session, parentAuthority);
			reattach(session, authority);

			if (parentAuthority != null) {

				subSegment = DbSubSegment.ByParentAndName(session, (DbAuthority) parentAuthority, subSegmentName);
				if (subSegment != null) throw new StoreDuplicateEntryException("Subsegment exists already.");
			}

			if (authority != null && parentAuthority != null &&
					authority.equals(parentAuthority)) authority = parentAuthority; 

			subSegment = new DbSubSegment();
			subSegment.setParent((DbAuthority) parentAuthority);
			subSegment.setName(subSegmentName);
			subSegment.setAuthority((DbAuthority) authority);

			if (parentAuthority != null) {

				((DbAuthority) parentAuthority).getChildren().add(subSegment);
			}

			if (authority != null) {

				((DbAuthority) authority).getSubSegments().add(subSegment);
			}

			session.save(subSegment);
			session.flush();
			session.refresh(subSegment);
			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw ex;
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegment);
	}

	public synchronized void releaseSubSegment(
			SubSegment subSegment)
	throws StoreException {

		log.trace("releaseSubsegment()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// delete subsegment

		try {

			session.beginTransaction();

			session.delete(subSegment);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	/*
	 * Here comes the implementation for StoreLookup.
	 */

	public synchronized SubSegment findRootSubSegment(
			String namespace) 
	throws StoreException {

		log.trace("findRootSubSegment()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// find root subsegment

		DbSubSegment rootSubSegment;

		try {

			session.beginTransaction();

			rootSubSegment = DbSubSegment.RootByName(session, namespace);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(rootSubSegment);
	}

	public synchronized SubSegment findSubSegment(
			Authority parentAuthority,
			String name) 
	throws StoreException {

		log.trace("findSubSegment()");

		// check parent authority

		if (parentAuthority == null) throw new NullPointerException();
		if (! (parentAuthority instanceof DbAuthority)) throw new StoreNotFoundException("Parent authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// find subsegment

		DbSubSegment subSegment;

		try {

			session.beginTransaction();

			reattach(session, parentAuthority);
			subSegment = DbSubSegment.ByParentAndName(session, (DbAuthority) parentAuthority, name);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegment);
	}

	public synchronized Authority getSubSegmentParentAuthority (
			SubSegment subSegment)
	throws StoreException {

		log.trace("getSubSegmentParentAuthority()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get parent authority

		DbAuthority parentAuthority;

		try {

			session.beginTransaction();

			reattach(session, subSegment);
			parentAuthority = ((DbSubSegment)subSegment).getParent();
			if (parentAuthority != null) Hibernate.initialize(parentAuthority);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(parentAuthority);
	}

	public synchronized Authority getSubSegmentAuthority (
			SubSegment subSegment)
	throws StoreException {

		log.trace("getSubSegmentAuthority()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get authority

		DbAuthority authority;

		try {

			session.beginTransaction();

			reattach(session, subSegment);
			authority = ((DbSubSegment)subSegment).getAuthority();
			if (authority != null) Hibernate.initialize(authority);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authority);
	}

	/*
	 * Here comes the implementation for StoreResettable, which provides a method for deleting all store contents.
	 */

	public synchronized void resetStore() throws StoreException {

		log.trace("resetStore()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// delete all subsegments and authorities

		try {

			session.beginTransaction();

			DbSubSegment.DeleteAll(session);
			DbAuthority.DeleteAll(session);

			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
	}

	/*
	 * Here comes the implementation for StoreBetterLookup, which provides advanced methods
	 * for getting information out of a store.
	 */

	public synchronized Authority[] listAuthorities() 
	throws StoreException {

		log.trace("listAuthorities()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbAuthority> authorities;

		try {

			session.beginTransaction();

			authorities = DbAuthority.All(session);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// build result list

		Authority[] result = new ArrayList<Authority> (authorities).toArray(new Authority[authorities.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized SubSegment[] listSubSegments() 
	throws StoreException {

		log.trace("listSubSegments()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all subsegments

		List<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			subSegments = DbSubSegment.All(session);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// build result list

		SubSegment[] result = new ArrayList<SubSegment> (subSegments).toArray(new SubSegment[subSegments.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized SubSegment[] getAuthoritySubSegments(
			Authority authority) 
	throws StoreException {

		log.trace("getAuthoritySegments()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get subsegments that lead to this authority

		Set<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			reattach(session, authority);
			subSegments = ((DbAuthority)authority).getSubSegments();
			Hibernate.initialize(subSegments);

			if (subSegments == null) throw new StoreNotFoundException("Subsegments do not exist.");

			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// build result list

		SubSegment[] result = new ArrayList<SubSegment> (subSegments).toArray(new SubSegment[subSegments.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized SubSegment[] getAuthorityChildSubSegments(
			Authority parent) 
	throws StoreException {

		log.trace("getAuthorityChildSegments()");

		// check authority

		if (parent == null) throw new NullPointerException();
		if (! (parent instanceof DbAuthority)) throw new StoreNotFoundException("Parent authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get subsegments that have this authority as parent

		Set<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			reattach(session, parent);
			subSegments = ((DbAuthority)parent).getChildren();
			Hibernate.initialize(subSegments);

			if (subSegments == null) throw new StoreNotFoundException("Subsegments do not exist.");

			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// build result list

		SubSegment[] result = new ArrayList<SubSegment> (subSegments).toArray(new SubSegment[subSegments.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized SubSegment[] getSynonymSubSegments(
			Authority parent,
			Authority authority) 
	throws StoreException {

		log.trace("getSynonymSegments()");

		// check authorities

		if (parent == null) throw new NullPointerException();
		if (! (parent instanceof DbAuthority)) throw new StoreNotFoundException("Parent authority is not from this store.");

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get subsegments that have the given parent authority and authority

		List<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			reattach(session, parent);
			reattach(session, authority);

			subSegments = DbSubSegment.ByParentAndAuthority(session, (DbAuthority) parent, (DbAuthority) authority);
			Hibernate.initialize(subSegments);

			if (subSegments == null) throw new StoreNotFoundException("Subsegments do not exist.");

			session.getTransaction().commit();
		} catch (StoreException ex) {

			log.error("", ex);
			rollback(session);
			throw(ex);
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// build result list

		SubSegment[] result = new ArrayList<SubSegment> (subSegments).toArray(new SubSegment[subSegments.size()]);

		// done

		log.trace("Done.");
		return(result);
	}

	public synchronized String[] getAuthorityQxris(Authority authority) throws StoreException {

		return(this.getAuthorityQxris(authority, true, true));
	}

	public synchronized String[] getAuthorityQxris(Authority authority, boolean reassignable, boolean persistent) throws StoreException {

		log.trace("getAuthorityQxris()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// recursively get full names for the authority;

		List<String> result;

		try {

			session.beginTransaction();

			reattach(session, authority);
			result = this.getAuthorityQxris(session, (DbAuthority) authority, 1, reassignable, persistent);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(result.toArray(new String[result.size()]));
	}

	private synchronized List<String> getAuthorityQxris(Session session, DbAuthority authority, int depth, boolean reassignable, boolean persistent) throws StoreException {

		// prevent too deep recursion (probably recursive aliases; maybe should find a better way of doing this

		if (depth > DatabaseStore.MAX_ALIAS_RECURSION) return(new ArrayList<String> ());

		ArrayList<String> qxris = new ArrayList<String> ();
		Set<DbSubSegment> subSegments;

		// if this is a root authority, we're done

		if (authority == null) {

			qxris.add("");
			return(qxris);
		}

		// find subsegments that have us as authority

		reattach(session, authority);

		subSegments = authority.getSubSegments();

		// the recursive result is the full names of the found subsegments' parents authorities, plus the local names

		for (DbSubSegment subSegment : subSegments) {

			List<String> rets = getAuthorityQxris(session, subSegment.getParent(), depth+1, reassignable, persistent);

			for (String ret : rets) {

				String subSegmentName = subSegment.getName();

				if ((! reassignable) && (! subSegmentName.contains("!"))) continue;
				if ((! persistent) && (subSegmentName.contains("!"))) continue;

				qxris.add(ret + subSegment.getName());
			}
		}

		if (qxris.equals("")) qxris.add("");

		// done

		return(qxris);
	}

	public synchronized Authority localLookup(
			XRIAuthority authorityPath) 
	throws StoreException {

		log.trace("localLookup()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// try to resolve the authority (must be in one of our root authorities)

		DbAuthority authority = null;

		try {

			session.beginTransaction();

			// find the root subsegment first

			LinkedList<String> subSegmentNames = new LinkedList<String> ();

			while(authorityPath != null) {

				DbSubSegment rootSubSegment = DbSubSegment.RootByName(session, authorityPath.toString());
				if (rootSubSegment != null) {

					authority = rootSubSegment.getAuthority(); 
					break;
				}

				log.trace(authorityPath.toString());

				// remove and remember the last subsegment one by one until we find a root subsegment 

				if (authorityPath.getLastSubSegment() != null) subSegmentNames.addFirst(authorityPath.getLastSubSegment().toString());

				authorityPath = authorityPath.getParent();
			}

			if (authority == null) return(null);

			// now iterate through the remaining subsegments to resolve the final authority.
			// from now on, we require that all subsegments are present in the store.
			// if we have no remaining subsegments at this point, we were given a root subsegment in the first place; which is also ok

			for (String subSegmentName : subSegmentNames) {

				// using the parent authority and subsegment name, find the next authority

				DbSubSegment subSegment;

				subSegment = DbSubSegment.ByParentAndName(session, authority, subSegmentName);
				if (subSegment == null) return(null);

				authority = subSegment.getAuthority();
			}

			Hibernate.initialize(authority);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authority);
	}

	/*
	 * Here comes the implementation for StoreAttributable, which provides methods for associating and reading
	 * contexts with authorities and subsegments (e.g. registration data, name of owner, etc.)
	 */

	public synchronized void setStoreAttributes(
			Map<String, String> attributes) throws StoreException {

		log.trace("setStoreAttributes()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update store data

		try {

			session.beginTransaction();

			DbStoreData storeData = DbStoreData.Singleton(session);
			storeData.setAttributes(attributes);

			session.update(storeData);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
	}

	public synchronized Map<String, String> getStoreAttributes() throws StoreException {

		log.trace("getStoreAttributes()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get value from store data

		Map<String, String> attributes;

		try {

			session.beginTransaction();

			attributes = DbStoreData.Singleton(session).getAttributes();

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(attributes);
	}

	public synchronized void setAuthorityAttributes(
			Authority authority, 
			Map<String, String> attributes) throws StoreException {

		log.trace("setAuthorityAttributes()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update authority

		try {

			session.beginTransaction();

			((DbAuthority) authority).setAttributes(attributes);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
	}

	public synchronized Map<String, String> getAuthorityAttributes(
			Authority authority)
			throws StoreException {

		log.trace("getAuthorityAttributes()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// get attributes from authority

		Map<String, String> attributes = ((DbAuthority) authority).getAttributes();

		// done

		log.trace("Done.");
		return(attributes);
	}

	public synchronized void setAuthorityIndex(
			Authority authority, 
			String indx) throws StoreException {

		log.trace("setAuthorityIndex(" + indx + ")");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update authority

		try {

			session.beginTransaction();

			reattach(session, authority);
			((DbAuthority) authority).setIndx(indx);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public synchronized String getAuthorityIndex(
			Authority authority)
	throws StoreException {

		log.trace("getAuthorityIndex()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// get index from authority

		String indx = ((DbAuthority) authority).getIndx();

		// done

		log.trace("Done.");
		return(indx);
	}

	public synchronized void setSubSegmentAttributes(
			SubSegment subSegment, 
			Map<String, String> attributes) throws StoreException {

		log.trace("setSubSegmentAttributes()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update subsegment

		try {

			session.beginTransaction();

			((DbSubSegment) subSegment).setAttributes(attributes);

			session.update(subSegment);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public synchronized Map<String, String> getSubSegmentAttributes(
			SubSegment subSegment)
			throws StoreException {

		log.trace("getSubSegmentAttributes()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// get attributes from subsegment

		Map<String, String> attributes = ((DbSubSegment)subSegment).getAttributes();

		// done

		log.trace("Done.");
		return(attributes);
	}

	public synchronized void setSubSegmentIndex(
			SubSegment subSegment, 
			String indx) throws StoreException {

		log.trace("setSubSegmentIndex(" + indx + ")");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update subsegment

		try {

			session.beginTransaction();

			reattach(session, subSegment);
			((DbSubSegment) subSegment).setIndx(indx);

			session.update(subSegment);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public synchronized String getSubSegmentIndex(
			SubSegment subSegment)
	throws StoreException {

		log.trace("getSubSegmentIndex()");

		// check subsegment

		if (subSegment == null) throw new NullPointerException();
		if (! (subSegment instanceof DbSubSegment)) throw new StoreNotFoundException("Subsegment is not from this store.");

		// get index from subsegment

		String indx = ((DbSubSegment) subSegment).getIndx();

		// done

		log.trace("Done.");
		return(indx);
	}

	public synchronized Authority[] listAuthoritiesByAttributes(
			Selector selector)
	throws StoreException {

		log.trace("listAuthoritiesByAttributes()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbAuthority> authorities;
		List<Authority> selectedAuthorities = new ArrayList<Authority> ();

		try {

			session.beginTransaction();

			authorities = DbAuthority.All(session);

			// build result list, using the selector

			for (DbAuthority Authority : authorities) {

				if (selector.select(Authority.getAttributes())) selectedAuthorities.add(Authority);
			}

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(selectedAuthorities.toArray(new Authority[selectedAuthorities.size()]));
	}

	public synchronized SubSegment[] listSubSegmentsByAttributes(
			Selector selector)
	throws StoreException {

		log.trace("listSubSegmentsByAttributes()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all subsegments

		List<DbSubSegment> subSegments;
		List<SubSegment> selectedSubSegments = new ArrayList<SubSegment> ();

		try {

			session.beginTransaction();

			subSegments = DbSubSegment.All(session);

			// build result list, using the selector

			for (DbSubSegment subSegment : subSegments) {

				if (selector.select(subSegment.getAttributes())) selectedSubSegments.add(subSegment);
			}

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(selectedSubSegments.toArray(new SubSegment[selectedSubSegments.size()]));
	}

	public synchronized Authority[] listAuthoritiesByAttributeValue(
			String key,
			String value)
	throws StoreException {

		log.trace("listAuthoritiesByAttributeValue(" + key + ", " + value + ")");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbAuthority> authorities;

		try {

			session.beginTransaction();

			authorities = DbAuthority.ByAttributeValue(session, key, value);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authorities.toArray(new Authority[authorities.size()]));
	}

	public synchronized SubSegment[] listSubSegmentsByAttributeValue(
			String key,
			String value)
	throws StoreException {

		log.trace("listSubSegmentsByAttributeValue(" + key + ", " + value + ")");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			subSegments = DbSubSegment.ByAttributeValue(session, key, value);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegments.toArray(new SubSegment[subSegments.size()]));
	}

	public synchronized Authority[] listAuthoritiesByIndex(
			String indx)
	throws StoreException {

		log.trace("listAuthoritiesByIndex(" + indx + ")");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbAuthority> authorities;

		try {

			session.beginTransaction();

			authorities = DbAuthority.AllByIndx(session, indx);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authorities.toArray(new Authority[authorities.size()]));
	}

	public synchronized SubSegment[] listSubSegmentsByIndex(
			String indx)
	throws StoreException {

		log.trace("listSubSegmentsByIndex(" + indx + ")");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all subsegments

		List<DbSubSegment> subSegments;

		try {

			session.beginTransaction();

			subSegments = DbSubSegment.AllByIndx(session, indx);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(subSegments.toArray(new SubSegment[subSegments.size()]));
	}

	/*
	 * Here comes the implementation for StoreEditable, which has methods for reading and manipulating
	 * services/internals/externals of subsegments.
	 */

	public synchronized void setXrd(
			Authority authority, 
			XRD xrd) throws StoreException {

		log.trace("setXrd()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// set XRD

		try {

			session.beginTransaction();

			((DbAuthority) authority).setXrd(xrd);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	/*
	 * Here comes the implementation for StoreMountable
	 */

	public void setAuthorityPath(
			Authority authority, 
			String path) throws StoreException {

		log.trace("mount(" + path + ", " + authority.getId() + ")");

		// adjust path

		while (path != null && path.startsWith("/")) path = path.substring(1);

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// update authority

		try {

			session.beginTransaction();

			reattach(session, authority);
			((DbAuthority) authority).setPath(path);

			session.update(authority);
			session.flush();
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		log.trace("Done.");
	}

	public String getAuthorityPath(
			Authority authority) throws StoreException {

		log.trace("getAuthorityPath()");

		// check authority

		if (authority == null) throw new NullPointerException();
		if (! (authority instanceof DbAuthority)) throw new StoreNotFoundException("Authority is not from this store.");

		// get path from authority

		String path = ((DbAuthority) authority).getPath();

		// done

		log.trace("Done.");
		return(path);
	}

	public Authority[] listAuthoritiesByPath(
			String path) throws StoreException {

		log.trace("listAuthoritiesByPath(" + path + ")");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get all authorities

		List<DbAuthority> authorities;

		try {

			session.beginTransaction();

			authorities = DbAuthority.AllByPath(session, path);

			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(authorities.toArray(new Authority[authorities.size()]));
	}

	/*
	 * Here comes the implementation for StoreStatistics
	 */

	public synchronized Long getAuthorityCount() throws StoreException {

		log.trace("getAuthorityCount()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get authority count

		Long count;

		try {

			session.beginTransaction();

			count = DbAuthority.Count(session);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(count);
	}

	public synchronized Long getSubSegmentCount() throws StoreException {

		log.trace("getSubSegmentCount()");

		// start database session

		Session session = this.getSessionFactory().getCurrentSession();

		// get subsegment count

		Long count;

		try {

			session.beginTransaction();

			count = DbSubSegment.Count(session);
			session.getTransaction().commit();
		} catch (Exception ex) {

			log.error("", ex);
			rollback(session);
			throw new StoreInternalException(ex, "Cannot access database.");
		}

		// done

		log.trace("Done.");
		return(count);
	}

	private static void reattach(Session session, Object object) {

		if (object == null || session.contains(object)) return;

		session.lock(object, LockMode.READ);
	}

	private static void rollback(Session session) {

		try {

			if (session.isOpen() && session.getTransaction().isActive()) session.getTransaction().rollback();
		} catch (Exception ex) {

			log.warn("", ex);
		}
	}
}
