package org.openxri.store.impl.db;

import java.io.ByteArrayInputStream;
import java.io.Serializable;
import java.io.StringReader;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.usertype.UserType;
import org.openxri.util.DOMUtils;
import org.openxri.xml.XRD;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

/**
 * A custom type for Hibernate to store instances of org.openxri.xml.XRD.
 * Stores the XRD by serializing its XML and putting it into a CLOB / TEXT column;
 * 
 * @author =peacekeeper
 */
public class XRDType implements UserType {

	private static Log log = LogFactory.getLog(XRDType.class.getName());

	private DocumentBuilder builder;

	public XRDType() throws HibernateException {

		log.trace("XRDType()");

		try {

			this.builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch (ParserConfigurationException ex) {

			throw new HibernateException("Cannot initialize XRD type.", ex);
		}
	}

	public Object assemble(Serializable cached, Object owner) throws HibernateException {

		log.trace("assemble()");

		if (cached == null) return(null);

		XRD xrd;
		String xrdText = (String) cached;

		try {

			Document document = this.builder.parse(xrdText);
			Element element = document.getDocumentElement();
			xrd = new XRD(element, false);
		} catch (Exception ex) {

			throw new HibernateException("Cannot assemble xrd.", ex);
		}

		return(xrd);
	}

	public Serializable disassemble(Object value) throws HibernateException {

		log.trace("disassemble()");

		if (value == null) return(null);

		XRD xrd = (XRD) value;
		String xrdText;

		try {

			Document doc = this.builder.newDocument();
			Node element = xrd.toDOM(doc);
			doc.appendChild(element);
			xrdText = DOMUtils.toString((Element) element, true, true);
		} catch (Exception ex) {

			throw new HibernateException("Cannot disassemble xrd.", ex);
		}

		return(xrdText);
	}

	public Object deepCopy(Object value) throws HibernateException {

		log.trace("deepCopy()");
		
		if (value == null) return(null);

		try {

			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document document = builder.parse(new InputSource(new StringReader(((XRD) value).toString())));
			
			XRD copy = new XRD();
			copy.fromDOM(document.getDocumentElement());
			return(copy);
		} catch(Exception ex) {

			throw new HibernateException(ex);
		}
	}

	public boolean equals(Object x, Object y) throws HibernateException {

		log.trace("equals()");

		if (x == null || y == null) return(false);
		if (x == y) return(true);

		XRD xrd1 = (XRD) x;
		XRD xrd2 = (XRD) y;

		boolean result = (xrd1.toString().equals(xrd2.toString()));

		log.trace("Done: " + result);
		return(result);	
	}

	public int hashCode(Object x) throws HibernateException {

		log.trace("hashCode()");

		if (x == null) return(0);

		XRD xrd = (XRD) x;

		return(xrd.hashCode());
	}

	public boolean isMutable() {

		log.trace("isMutable()");

		return(true); 
	}

	public Object nullSafeGet(ResultSet rs, String[] names, Object owner) 
	throws HibernateException, SQLException {

		log.trace("nullSafeGet()");

		XRD xrd;
		String str = rs.getString(names[0]);

		try {

			if (str != null) {

				Document document = this.builder.parse(new ByteArrayInputStream(str.getBytes()));
				Element element = document.getDocumentElement();

				try {

					xrd = new XRD(element, false);
				} catch (Exception ex) {

					log.error(ex);
					throw(ex);
				}
			} else {

				xrd = null;
			}
		} catch (Exception ex) {

			throw new HibernateException("Cannot retrieve XRD.", ex);
		}

		return(xrd);
	}

	public void nullSafeSet(PreparedStatement st, Object value, int index)
	throws HibernateException, SQLException {

		log.trace("nullSafeSet()");

		XRD xrd = (XRD) value;

		try {

			if (xrd != null) {

				Document doc = this.builder.newDocument();
				Node element;

				try {

					element = xrd.toDOM(doc);
				} catch (Exception ex) {

					log.error(ex);
					throw(ex);
				}

				doc.appendChild(element);
				String providerIdText = DOMUtils.toString((Element) element, true, true);
				st.setString(index, providerIdText);
			} else {

				st.setString(index, null);
			}
		} catch (Exception ex) {

			throw new HibernateException("Cannot store xrd.", ex);
		}
	}

	public Object replace(Object original, Object target, Object owner) throws HibernateException {

		log.trace("replace()");

		if (original == null) return(null);

		XRD xrd = (XRD) original;

		return (xrd.clone());
	}

	public Class<?> returnedClass() {

		log.trace("returnedClass()");

		return(org.openxri.xml.XRD.class);
	}

	public int[] sqlTypes() {

		log.trace("sqlTypes()");

		return(new int[] { Types.CLOB });
	}
}
