package org.openxri.urimapper.impl;

import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.openxri.config.impl.AbstractComponent;
import org.openxri.urimapper.URIMapper;
import org.openxri.urimapper.URIMapperRequest;
import org.openxri.urimapper.URIMapperResult;


/**
 * This abstract class can parse an URI using regular expressions for the root namespace (e.g. @free)
 * and the query (*earth*moon). Subclasses need to provide those regular expressions in the constructor
 * and implement the completeURI method of the URIMapper interface.
 * 
 * @author =peacekeeper
 */
public abstract class RegexURIMapper extends AbstractComponent implements URIMapper {

	protected Pattern namespacePattern;			// must return one group with the namespace (e.g. @free)
	protected Pattern queryPattern;				// must return one group with the query (e.g. *earth*moon)

	protected RegexURIMapper(Properties properties, Pattern namespacePattern, Pattern queryPattern) {

		super(properties);
		
		this.namespacePattern = namespacePattern;
		this.queryPattern = queryPattern;
	}

	public URIMapperResult parseRequest(URIMapperRequest request) {

		String requestPath = request.getPath();
		String requestQuery = request.getQuery();

		String resultNamespace = null;
		String resultQuery = null;

		String uri = requestPath + (requestQuery != null ? "?" + requestQuery : "");

		Matcher namespaceMatcher = this.namespacePattern.matcher(uri);
		Matcher queryMatcher = this.queryPattern.matcher(uri);

		if (! namespaceMatcher.matches() || ! queryMatcher.matches()) return(null);
		
		// read the namespace

		if (namespaceMatcher.groupCount() == 1) {

			resultNamespace = namespaceMatcher.group(1);
		}

		// need either a namespace or an id

		if (queryMatcher.groupCount() == 1) {

			resultQuery = queryMatcher.group(1);
		}

		URIMapperResult uriMapperResult = new URIMapperResult(resultNamespace, resultQuery);

		return(uriMapperResult);
	}

	public Pattern getNamespacePattern() {

		return (this.namespacePattern);
	}

	public void setNamespacePattern(Pattern nsPattern) {

		this.namespacePattern = nsPattern;
	}

	public Pattern getQueryPattern() {

		return (this.queryPattern);
	}

	public void setQueryPattern(Pattern queryPattern) {

		this.queryPattern = queryPattern;
	}
}
