package org.openxri.store;

import java.util.Date;
import java.util.Properties;

import junit.framework.TestCase;

import org.openxri.GCSAuthority;
import org.openxri.XRI;
import org.openxri.XRISubSegment;
import org.openxri.store.impl.db.DatabaseStore;
import org.openxri.xml.XRD;

/**
 * This test case shows how the server's Store could be used by
 * a resolver as a cache.
 * 
 * @author peace-keeper
 */
public class StoreCacheTest extends TestCase {

	private DatabaseStore store;

	public void setUp() throws Exception {

		Properties properties = new Properties();
		properties.load(this.getClass().getResourceAsStream("store.properties"));

		store = new DatabaseStore(properties);
		store.init();

		// we clear everything from the store

		store.resetStore();
	}

	public void tearDown() throws Exception {

		store.close();
	}

	public void testCache() throws Exception {

		XRI xri;
		XRD xrd;

		// Let's say we want to resolve @free*earth*moon

		xri = new XRI("@free*earth*moon");
		GCSAuthority xriAuthority = (GCSAuthority) xri.getAuthorityPath();

		// We initialize the store by creating a root subsegment with an empty XRD (i.e. an entry point)

		SubSegment storeSubSegment = store.createRootSubSegment(xriAuthority.getGCSRoot(), new XRD());
		Authority storeAuthority = store.getSubSegmentAuthority(storeSubSegment);

		// We iterate through all subsegments of the XRI that we were given to resolve.

		for (int i=0; i<xriAuthority.getNumSubSegments(); i++) {

			XRISubSegment subSegment = xriAuthority.getSubSegmentAt(i);

			// Somehow we obtain an XRD for this subsegment

			xrd = magicXriResolution(subSegment);

			// Now we cache the XRD we just got. We do this by creating a new subsegment in the store.

			storeSubSegment = store.registerSubsegment(storeAuthority, subSegment.toString(), xrd);
			storeAuthority = store.getSubSegmentAuthority(storeSubSegment);
		}

		// Let's say a while later we want to resolve @free*earth*moon again... We find it in the store, yay!!!

		xri = new XRI("@free*earth*moon");
		xriAuthority = (GCSAuthority) xri.getAuthorityPath();

		storeAuthority = store.localLookup(xriAuthority);
		assertNotNull(storeAuthority);

		xrd = storeAuthority.getXrd();
		assertNotNull(xrd);

		assertEquals(xrd.getQuery(), "*moon");

		// We could also read the cache step by step, if we want to get the XRD for every subsegment.

		storeSubSegment = store.findRootSubSegment(xriAuthority.getGCSRoot());
		storeAuthority = store.getSubSegmentAuthority(storeSubSegment);

		for (int i=0; i<xriAuthority.getNumSubSegments(); i++) {

			XRISubSegment subSegment = xriAuthority.getSubSegmentAt(i);

			// Find the authority in the store for the current subsegment

			storeSubSegment = store.findSubSegment(storeAuthority, subSegment.toString());
			storeAuthority = store.getSubSegmentAuthority(storeSubSegment);

			assertNotNull(storeAuthority);
			xrd = storeAuthority.getXrd();
			assertNotNull(xrd);
			assertEquals(xrd.getQuery(), subSegment.toString());
		}
	}

	/**
	 * This helper method simulates the actual resolution process. It returns an XRD for a subsegment.
	 */
	public static XRD magicXriResolution(XRISubSegment subSegment) {

		XRD xrd;

		xrd = new XRD();
		xrd.setQuery(subSegment.toString());
		xrd.setExpires(new Date(new Date().getTime() + 3600000));

		return(xrd);
	}
}
